;;; -*- emacs-lisp -*-

;;; This file is an addition to the Insidious Big Brother Database
;;; (aka BBDB), copyright (c) 1991, 1992 Jamie Zawinski
;;; <jwz@jwz.org>.
;;; 
;;; Copyright (C) 1999 Sen Nagata <sen@eccosys.com>
 
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 2 of
;;; the License.
          
;;; This program is distributed in the hope that it will be
;;; useful, but WITHOUT ANY WARRANTY; without even the implied
;;; warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
;;; PURPOSE.  See the GNU General Public License for more details.
          
;;; You should have received a copy of the GNU General Public
;;; License along with this program; if not, write to the Free
;;; Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
;;; MA 02111-1307 USA

;;; This file was heavily based on bbdb-ftp.el by Ivan Vazquez 
;;; <ivan@haldane.bu.edu> 

;;; This file adds the ability to define mailing lists in a BBDB, much the same
;;; way one adds a regular person's name to the BBDB.  See RFC 2369.

;;; Note that Mailing List BBDB entries differ from regular entries by the
;;; fact that the Name Field must have the mailing list name preceeded by the
;;; bbdb-mailing-list-name-designator-prefix.  This defaults to 
;;; "Mailing List:" BBDB Mailing List entries also have some new fields added:
;;;
;;;   ml-type
;;;   ml-help
;;;   ml-unsubscribe
;;;   ml-subscribe
;;;   ml-post
;;;   ml-owner
;;;   ml-archive
;;; 
;;; These are added to the notes alist part of the bbdb-records, the original 
;;; bbdb-record structure remains untouched.

;;; The following user-level commands are defined for use:
;;;

;;; bbdb-create-mailing-list -
;;;            Add a new mailing-list entry to the bbdb database; prompts
;;;            for all relevant info using the echo area, inserts the
;;;            new record in the db, sorted alphabetically.

;;; TODO: implement the following:

;;; The package can be installed by compiling and adding the following
;;; line to your .emacs.

;;; (autoload 'bbdb-create-mailing-list "bbdb-ml" "Mailing List BBDB Package" t)

(require 'bbdb)

(defconst bbdb-ml-version "bbdb-ml.el 0.1")

;; based this on something in bbdb.el
(defgroup bbdb-utilities-ml nil
  "Customizations for using mailing lists stored in BBDB records."
  :group 'bbdb-utilities)
(put 'bbdb-utilities-ml 'custom-loads '("bbdb-ml"))

(defcustom bbdb-default-mailto-command-prefix "mailto:"
  "*The prefix for a mailing list mailto url command."
  :group 'bbdb-utilities-ml
  :type 'string)

(defcustom bbdb-default-http-command-prefix "http://"
  "*The prefix for a mailing list http url command."
  :group 'bbdb-utilities-ml
  :type 'string)

(defcustom bbdb-mailing-list-name-designator-prefix "Mailing List: "
  "*The prefix that all mailing lists in the bbdb will have in their name 
field."
  :group 'bbdb-utilities-ml
  :type 'string)

(defmacro defun-bbdb-raw-notes-accessor (slot)
  "Expands into an accessor function for slots in the notes alist."
  (let ((fn-name (intern (concat "bbdb-record-" (symbol-name slot)))))
    (list 'defun fn-name (list 'record)
	  (list 'cdr 
		(list 'assoc (list 'quote slot)
		      (list 'bbdb-record-raw-notes 'record))))))

;; (defun-bbdb-raw-notes-accessor ml-type) expands to:
;;
;; (defun bbdb-record-ml-type (record)
;;   (cdr (assoc (quote ml-type) (bbdb-record-raw-notes record))))

(defvar bbdb-ml-field-prefix "ml-")

(defvar bbdb-ml-notes-action-name-list
  '("help"
    "unsubscribe"
    "subscribe"
    "post"
    "owner"
    "archive")
  "List of actions for mailing list records.")

;; check passed value?
(defun bbdb-ml-action-name-to-symbol (action-name)
  "Return symbol corresponding to ACTION-NAME."
  (intern (concat bbdb-ml-field-prefix (downcase action-name))))

;; check passed value?
(defun bbdb-ml-symbol-to-action-name (symbol)
  "Return action name corresponding to SYMBOL."
  (substring (symbol-name symbol)
	     (length bbdb-ml-field-prefix)))

;; check passed value?
(defun bbdb-ml-action-name-to-symbol-name (action-name)
  "Return symbol name corresponding to ACTION-NAME."
  (symbol-name 
   (intern 
    (concat bbdb-ml-field-prefix (downcase action-name)))))

;; check passed value?
(defun bbdb-ml-symbol-name-to-action-name (symbol-name)
  "Return action name corresponding to SYMBOL-NAME."
  (substring symbol-name
	     (length bbdb-ml-field-prefix)))

(defvar bbdb-ml-notes-fieldname-alist
  (cons (cons "ml-type" nil)
	(mapcar
	 (lambda (x)
	   (cons
	    (bbdb-ml-action-name-to-symbol-name x)
	    nil))
	 bbdb-ml-notes-action-name-list))
  "List of notes fields for mailing list records.")
	
; TODO: can't we do the following programmatically based on some list?
(defun-bbdb-raw-notes-accessor ml-type)
(defun-bbdb-raw-notes-accessor ml-help) 
(defun-bbdb-raw-notes-accessor ml-unsubscribe) 
(defun-bbdb-raw-notes-accessor ml-subscribe) 
(defun-bbdb-raw-notes-accessor ml-post) 
(defun-bbdb-raw-notes-accessor ml-owner) 
(defun-bbdb-raw-notes-accessor ml-archive) 

(defun bbdb-record-mailing-list (record)
  "Acessor Function.  Returns the mailing-list field of the BBDB record or 
nil."
  (let* ((name (bbdb-record-name record))
	 (ml-pfx-regexp (concat bbdb-mailing-list-name-designator-prefix " *"))
	 (mailing-list 
	  (and (string-match ml-pfx-regexp name) 
	       (substring name (match-end 0)))))
    mailing-list))

(defun remove-leading-whitespace (string)
  "Remove any spaces or tabs from only the start of the string."
  (let ((space-char-code (string-to-char " "))
	(tab-char-code ?\t)
	(index 0))
    (if string
	(progn 
	  (while (or (char-equal (elt string index) space-char-code)
		     (char-equal (elt string index) tab-char-code))
	    (setq index (+ index 1)))
	  (substring string index))
      nil)))

(defun bbdb-read-new-mailing-list-record ()
  "Prompt for and return a completely new bbdb-record that is
specifically a mailing list entry.  Doesn't insert it in to the database
or update the hashtables, but does insure that there will not be name
collisions."
  (bbdb-records) ; make sure database is loaded
  (if bbdb-readonly-p 
      (error "The Insidious Big Brother Database is read-only."))
  (let (mailing-list)
    (bbdb-error-retry
     (progn
       (setq mailing-list 
	     (bbdb-read-string "Mailing List: "))
       (setq mailing-list 
	     (concat bbdb-mailing-list-name-designator-prefix mailing-list))
       (if (bbdb-gethash (downcase mailing-list))
	    (error "%s is already in the database" mailing-list))))

    (let* ((ml-type 
	    (bbdb-read-string "ML type: "))

	   (ml-help 
	    (bbdb-read-string "Help URL: " 
			      bbdb-default-mailto-command-prefix))

	   (ml-unsubscribe 
	    (bbdb-read-string "Unsubscription URL: " 
			      bbdb-default-mailto-command-prefix))

	   (ml-subscribe 
	    (bbdb-read-string "Subscription URL: "
			      bbdb-default-mailto-command-prefix))

	   (ml-post 
	    (bbdb-read-string "Post URL: "
			      bbdb-default-mailto-command-prefix))

	   (ml-owner 
	    (bbdb-read-string "Owner URL: "
			      bbdb-default-mailto-command-prefix))

	   (ml-archive 
	    (bbdb-read-string "Archive URL: "
			      bbdb-default-http-command-prefix))

	   (company 
	    (bbdb-read-string "Company: "))

	   (notes 
	    (bbdb-read-string "Additional Comments: "))

	   (names (bbdb-divide-name mailing-list))
	   (firstname (car names))
	   (lastname (nth 1 names)))

      (if (string= company "") 
	  (setq company nil))
      (if (string= ml-type "") 
	  (setq ml-type nil))
      (if (or (string= ml-help bbdb-default-mailto-command-prefix) 
	      (string= ml-help ""))
	  (setq ml-help nil))
      (if (or (string= ml-unsubscribe bbdb-default-mailto-command-prefix) 
	      (string= ml-unsubscribe ""))
	  (setq ml-unsubscribe nil))
      (if (or (string= ml-subscribe bbdb-default-mailto-command-prefix) 
	      (string= ml-subscribe ""))
	  (setq ml-subscribe nil))
      (if (or (string= ml-post bbdb-default-mailto-command-prefix) 
	      (string= ml-post ""))
	  (setq ml-post nil))
      (if (or (string= ml-owner bbdb-default-mailto-command-prefix) 
	      (string= ml-owner ""))
	  (setq ml-owner nil))
      (if (or (string= ml-archive bbdb-default-http-command-prefix) 
	      (string= ml-archive ""))
	  (setq ml-archive nil))
      (if (string= notes "")
	  (setq notes nil))

      (let ((record
	     (vector firstname lastname nil company nil nil nil 
		     (append 
		      (if notes 
			  (list (cons 'notes notes)) nil)
		      (if ml-type 
			  (list (cons 'ml-type ml-type)) nil)
		      (if ml-help 
			  (list (cons 'ml-help ml-help)) nil)
		      (if ml-unsubscribe 
			  (list (cons 'ml-unsubscribe ml-unsubscribe)) nil)
		      (if ml-subscribe 
			  (list (cons 'ml-subscribe ml-subscribe)) nil)
		      (if ml-post
			  (list (cons 'ml-post ml-post)) nil)
		      (if ml-owner 
			  (list (cons 'ml-owner ml-owner)) nil)
		      (if ml-archive 
 			  (list (cons 'ml-archive ml-archive)) nil))
		     (make-vector bbdb-cache-length nil))))
	record))
    ))
   
(defun bbdb-create-mailing-list (record)
  "Add a new mailing-list entry to the bbdb database; prompts for all
relevant info using the echo area, inserts the new record in the db,
sorted alphabetically."
  (interactive (list (bbdb-read-new-mailing-list-record)))
  (bbdb-invoke-hook 'bbdb-create-hook record)
  (bbdb-change-record record t)
  (bbdb-display-records (list record)))

(provide 'bbdb-ml)
