;;;		mew.el :: Messaging in the Emacs World
;;;
;;;	      Copyright (C) 1994-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: March 22, 1994
;;; Revised: April 21, 1997
;;;
;;; The updated version is available from:
;;;	ftp://ftp.aist-nara.ac.jp/pub/elisp/Mew/mew-current.tar.gz
;;;	http://fukuda.aist-nara.ac.jp/~kazu/mew.html
;;;
;;; Minimum setup:
;;;	(autoload 'mew "mew" nil t)
;;;	(autoload 'mew-send "mew" nil t)
;;;	(setq mew-mail-domain-list '("your mail domain"))
;;;	(setq mew-icon-directory "icon directory")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Mew version
;;;

(defconst mew-version "Mew version 1.70")
(provide 'mew)
(require 'mew-env)

(defun mew-version-show ()
  "Show mew-version in minibuffer."
  (interactive)
  (message "%s" mew-version))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; For developers
;;;

(defvar mew-debug nil)
;(setq mew-debug nil)
;(setq mew-debug t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; User customizable variables. See also other ".el" files.
;;;

;;;
;;; Common
;;;

(defvar mew-profile (or (getenv "MH") (expand-file-name "~/.mh_profile")))

(defvar mew-mail-domain-list nil
  "*Your e-mail address domain list like 
(\"is.aist-nara.ac.jp\" \"csce.kyushu-u.ac.jp\").
They are used for mail-domain completion in draft mode(C-cTAB)")

(defvar mew-mail-domain (or (car mew-mail-domain-list) "")
  "*Your e-mail address domain.")

(defvar mew-mail-address-list nil
  "*The addresses included in this list never appear on the Cc:
field on a draft buffer.")

(cond 
 ((string= mew-mail-domain "")
  (defvar mew-mail-address (user-login-name)))
 (t
  (defvar mew-mail-address (concat (user-login-name) "@" mew-mail-domain)))
 )

(defvar mew-window-configuration 
  '((summary (1  0))
    (message (8 31))
    (draft   (1  0)))
  "*Ratio of windows")

;;;
;;; Summary mode
;;;

(defvar mew-analysis t
  "*MIME syntax analysis and cache when non-nil.
   Skip syntax analysis when nil.")

(defvar mew-cache-size 10
  "*Number of buffer for message cache.")

(defvar mew-summary-cache-file ".mew-cache"
  "*Cache file for summary mode contents.")

(defvar mew-summary-cache-use t
  "*If non-nil, insert mew-summary-cache-file into summary mode buffer 
when the summary mode is visited and it does not exist in buffer. 
mew-summary-cache-file is automatically saved on inc and scan.")

(defvar mew-window-use-full nil
  "*Dynamic window configuration when nil. This variable will be obsolated
because mew will support mew-window-percent(e.g 50, 100, etc.).")

(defvar mew-window-home-buffer "*scratch*"
  "*Buffer name to return if window stack is empty.")

(defvar mew-summary-show-direction 'down
  "*Direction for SPC at end of message. 
'up 'down 'next(current direction) 'stop")

(defvar mew-mime-compose-folder-delete 'ask
 "*Action to delete MIME draft directory after sending a multipart letter.
Proper value is 'ask', 'delete', and 'retain'.")

(defvar mew-summary-recenter-p t)
(defvar mew-break-pages t)
(defvar mew-page-delimiter "^\^L")

;;
;; Highlight
;;

(defvar mew-highlight-lines-use nil
  "*Use highlight line in summary mode.")

(defvar mew-underline-lines-use nil
  "*Use underline line in summary mode.")

(defvar mew-bold-url-size 3000
  "*Use bold font at url maximize size.
If nil, all contents are parsed.")

;;;
;;; Draft mode
;;;

(defvar mew-header-separator "----")
(defvar mew-eoh "^$")
(defvar mew-eoh2 (format "^\\(%s\\|\\)$" (regexp-quote mew-header-separator)))
(defvar mew-lwsp "^[ \t]")

(defvar mew-reply-to nil
  "*A value inserted into Reply-To: field in draft mode if non-nil.")

(defvar mew-fcc nil
  "*A value inserted into Fcc: field in draft mode if non-nil.")

(defvar mew-cc nil
  "*A value inserted into Cc: field in draft mode if non-nil.")

(defvar mew-dcc nil
  "*A value inserted into Dcc: field in draft mode if non-nil.")

(defvar mew-from nil
  "*A value inserted into From: field in draft mode if non-nil.")

(defvar mew-x-mailer
  (concat mew-version " on "
	  (if (string-match "\\(^[0-9.]*\\) XEmacs" emacs-version)
	      (concat "XEmacs " (substring emacs-version
					  (match-beginning 1)
					  (match-end 1)))
	    (concat "Emacs " emacs-version))
	  (and (boundp 'mule-version)
	       (string-match "^[0-9.]*" mule-version)
	       (concat " / Mule " (substring mule-version
					    (match-beginning 0) 
					    (match-end 0)))))
  "*A value inserted into From: field in draft mode if non-nil.")

(defvar mew-header-alist nil
  "*Alist of key and value pair for header field to be inserted on draft.
'((\"X-fingerprint:\" . \"6B 63 38 88 67 5E 96 8E  CE A4 62 73 3F 11 64 94\")
  (\"X-URL:\" . \"http://uranus.aist-nara.ac.jp/~kazu/\"))"
  )

(defvar mew-auto-add-content-type nil
  "*Automatically adds MIME Content-Type: header if non-nil, asks otherwise.")

(defvar mew-ask-cc t
  "*Prompt user for CC: field if non-nil")

(defvar mew-ask-subject nil
  "*Prompt user for Subject: field when send if non-nil")

(defvar mew-ask-range t
  "*If non-nil, ask for a range for scanning. If false, always
use default.")

(defvar mew-ask-newsgroups nil
  "*If non-nil, prompt user if he/she want to include Newsgroups: field")

;;;
;;; Signature
;;; 

(defvar mew-signature-file "~/.signature"
  "*A signature file to be inserted in Draft mode. To support multiple
signature files, use \"c-sig.el\".")

(defvar mew-signature-insert-last nil 
  "*If non-nil, the signature file is inserted in the last of body. 
Otherwise, it is inserted at the currect point. If you created multipart
and mew-signature-as-lastpart is non-nil, this variable is ignored.")

(defvar mew-signature-as-lastpart nil
  "*If non-nil, the signature file is appended as the final part 
when you create multipart.")

(defvar mew-signature-description "My signature"
  "*This variable is used as a description if the signature is appended
as the final part.")

;;;
;;; Citation
;;;

(defvar mew-cite-prefix "> "
  "*Prefix of citation.")
(defvar mew-cite-hook nil
  "*If you want to use super-cite, (setq mew-cite-hook 'sc-cite-original).")

(defvar mew-cite-prefix-function nil
  "*Function called on citations. A good candidate is 
'mew-cite-prefix-username")

(defvar mew-cite-prefix-confirmp nil
  "*If non-nil, ciattion prefix (such as \"kazu> \") is
confirmed to be used.")

(defun mew-cite-prefix-username ()
  "A good candidate for mew-cite-prefix-function.
The citation style is \"from_address> \", e.g. \"kazu> \""
  (format "%s> " (mew-header-delete-at
		  (mew-header-extract-addr
		   (mew-header-get-value "From:"))))
  )

(defvar mew-cite-fields '("From:" "Subject:" "Date:")
  "*The fields that you want to extract as citation label. 
If you change this valuable, you must change mew-cite-format.
The value of the first field becomes the first argment for mew-cite-format.
(e.g. The default first argment is a value of From: field.)
The value of the second field becomes the second argment for mew-cite-format.
....")

(defvar mew-cite-format "From: %s\nSubject: %s\nDate: %s\n\n"
  "*Format for the citation label.")

;;;
;;; Summary and Draft glue
;;;

;;
;; Temporary solution to decide To: when reply
;;

(defvar mew-noreplyto-to-list '("From:")
  "*To: field list picked at mew-summary-reply if Reply-To: doesn't exist.")

(defvar mew-noreplyto-cc-list '("To:" "Cc:" "Apparently-To:")
  "Cc: field list picked at mew-summary-reply if Reply-To: doesn't exist.")

(defvar mew-replyto-to-list '("Reply-To:" "From:")
  "*To: field list picked at mew-summary-reply if Reply-To: exists.")

(defvar mew-replyto-cc-list '("To:" "Cc:" "Apparently-To:")
  "*Cc: field list picked at mew-summary-reply if Reply-To: exists.")

;;;
;;; Misc
;;;

;;BBDB not fully supported
(defvar mew-use-bbdb nil
  "*Use Insidious Big Brother Database if t")

;; Icon
(defvar mew-multipart-icon-position 'right
  "*Position where multipart icons are displayed. 
If 'left, displayed at the left size of the default toolbar.
If 'right, displayed at the right size of the default toolbar.
Otherwise, not displayed.")

;;
;; X face
;;
;; Use highlight-headers-hack-x-face-p on XEmacs instead.
(defvar mew-x-face-filter '("uncompface" "ikon2xbm"))
(defvar mew-x-face-prog "xv")
(defvar mew-x-face-args nil)
(defvar mew-x-face-file "~/.xface"
  "*If non-nil and the file exists, X-Face: fields is inserted.")

;;;
;;; File
;;;

(defvar mew-inbox "+inbox"
  "*Mew's default folder")
(defvar mew-folders-file ".folders"
  "*A file to save folder list. It is used on start-up to
make start-up much faster.")
(defvar mew-folders-ignore '("from")
  "*Folder list to contain messages whose To: is you and From: is
personal, e.g. a friend. Directories under this folder is ignored
for refile guess.")
(defvar mew-folders-default-folder "from"
  "*A default folder used by mew-guess-by-default.
mew-guess-by-default append the \"From:\" value to it.
e.g. \"+from/kazu\"")

(defvar mew-file-append-p nil
  "*If non-nil, a message or a part is appended to the existing file
on mew-summary-save(\"y\"). Otherwise overwrited.")

(defvar mew-temp-file-initial (expand-file-name (user-login-name) "/tmp")
  "*Hint to make a secure directory on the local file system.
On setup phase Mew make a secure directory from this variable
and set mew-temp-file a file name prefex contained the directory name.
The directory must be unreadable from others, otherwise it might become 
a big security hole. Mew never uses 'call-process-region' rather does use
'call-process' creating a temporary file with mew-temp-file by itself.
If 'call-process-region' is used, Emacs creates a temporary file
(probably in /tmp). So bad guys can wiretap the temporary file.")

;(defvar mew-draft-mime-folder (format "+%s/mime" mew-draft-folder))
(defvar mew-draft-mime-folder "+drafts/mime"
  "A directory where attachments are contained.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Hooks
;;;

(defvar mew-init-hook nil
  "*Hook called at initialize time.")
(defvar mew-summary-mode-hook nil
  "*Hook called in summary mode.")
(defvar mew-draft-mode-hook nil
  "*Hook called in draft mode.")
(defvar mew-message-mode-hook nil
  "*Hook called in message mode.")
(defvar mew-message-hook nil
  "*Hook called whenever message displayed.")
(defvar mew-send-hook nil
  "*Hook called before sending a message")
(defvar mew-suspend-hook nil
  "*Hook called on suspend.")
(defvar mew-quit-hook nil
  "*Hook called on quit.")
(defvar mew-summary-inc-sentinel-hook nil
  "*Hook called when inc finished.")
(defvar mew-summary-scan-sentinel-hook nil
  "*Hook called when scan finished.")
(defvar mew-summary-exec-hook nil
  "*Hook called when mew-summary-exec finished.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Buffers
;;;

(defconst mew-buffer-message     "*Mew message*")
(defconst mew-buffer-mime       " *mew mime*")
(defconst mew-buffer-cache      " *mew cache*")
(defconst mew-buffer-watch       "*Mew watch*")
(defconst mew-buffer-tmp        " *mew tmp*")
(defconst mew-buffer-completions "*Mew completions*")
(defconst mew-buffer-whom        "*Mew whom*")

(defun mew-buffer-message ()
  (if window-system
      (concat
       mew-buffer-message
       (int-to-string
	(mew-member-match 
	 (cdr (assq
	       'window-id
	       (frame-parameters (window-frame (selected-window)))))
	 (sort
	  (mapcar
	   (function 
	    (lambda (frame)
	      (cdr (assq
		    'window-id
		    (frame-parameters frame)))))
	   (frame-list))
	  (function string<)))))
    mew-buffer-message
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Buffer local variables
;;;

(defvar mew-summary-buffer-process nil)
(defvar mew-summary-buffer-string nil)
(defvar mew-summary-buffer-direction nil)
(defvar mew-summary-buffer-folder-cache-time nil)
(defvar mew-summary-buffer-refile nil)
(defvar mew-summary-buffer-disp-msg t)
(defvar mew-summary-buffer-left-msgs "-")
(defvar mew-watch-buffer-process nil)
(defvar mew-draft-buffer-header nil)
(defvar mew-draft-buffer-attach nil)
(defvar mew-cache-attribute nil)
(defvar mew-decode-syntax nil)
(defvar mew-decode-error nil)
(defvar mew-encode-syntax nil)
(defvar mew-underline-overlay nil)

(mapcar
 (function make-variable-buffer-local)
 (list 'mew-summary-buffer-process
       'mew-summary-buffer-string
       'mew-summary-buffer-direction
       'mew-summary-buffer-folder-cache-time
       'mew-summary-buffer-refile
       'mew-summary-buffer-disp-msg
       'mew-summary-buffer-left-msgs
       'mew-watch-buffer-process
       'mew-draft-buffer-header
       'mew-draft-buffer-attach
       'mew-cache-attribute
       'mew-encode-syntax
       'mew-decode-syntax
       'mew-decode-error
       'mew-underline-overlay
       ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Sub-programs
;;;

(defvar mew-prog-inc         "inc")
(defvar mew-prog-scan        "scan")
(defvar mew-prog-vscan       "vscan")
(defvar mew-prog-ali         "ali")
(defvar mew-prog-folders     "folders")
(defvar mew-prog-folder      "folder")
(defvar mew-prog-pack        "folder")
(defvar mew-prog-rmm         "rmm")
(defvar mew-prog-refile      "refile")
(defvar mew-prog-send        "send")
(defvar mew-prog-sortm       "sortm")
(defvar mew-prog-pick        "pick")
(defvar mew-prog-msgchk      "msgchk")
(defvar mew-prog-mime-encode "mewencode")
(defvar mew-prog-mime-decode "mewdecode")
(defvar mew-prog-audio       (list "cat" "/dev/audio" mew-buffer-tmp nil))
;(defvar mew-prog-audio      '("arecord" nil nil nil "-silentlevel" "-100"))
(defvar mew-prog-unshar      "unshar")
(defvar mew-prog-uumerge     "uumerge")
(defvar mew-prog-tar         "tar")
(defvar mew-prog-compress    "compress")
(defvar mew-prog-gzip        "gzip")
(defvar mew-prog-shell       "/bin/sh")
;(defvar mew-prog-shell       "cmd.exe")

(defvar mew-prog-shell-arg   "-c")

(defvar mew-prog-inc-arg-list nil
  "*Argument list for mew-prog-inc")
(defvar mew-prog-scan-arg-list nil
  "*Argument list for mew-prog-scan")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Initialize global variables
;;;

(setplist 'mew-current-state 
	  (list
	   'message    nil 
	   'cache      nil
	   'part       nil
	   'window     nil
	   ))

(defun mew-current-get (key)
  (get 'mew-current-state key))

(defun mew-current-set (key value)
  (put 'mew-current-state key value))

(setq mew-draft-folder nil)
(setq mew-alias-alist  ())
(setq mew-folder-list  ())
(setq mew-folder-alist ())
(setq mew-refile-alist ())
(setq mew-folders (list mew-inbox))

(setq mew-message-citation nil)
(setq mew-message-citation-buffer nil) ;; should be nil

(setq mew-process-file-alist nil)

;; See mew-temp-file-initial.
(setq mew-temp-dir nil)  ;; the default is "/tmp/user_name_uniq"
(setq mew-temp-file nil) ;; the default is "/tmp/user_name_uniq/mew"

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Window configuration stack
;;;

(defvar mew-window-stack nil)

(defun mew-window-push ()
  (interactive)
  (let* ((key (if mew-emacs-p (selected-frame) 'dummy))
	 (stack (mew-alist-search mew-window-stack key)))
    (setq stack (cons (current-window-configuration) stack))
    (setq mew-window-stack
	  (mew-alist-add mew-window-stack key stack))
    ))

(defun mew-window-pop ()
  (interactive)
  (let* ((key (if mew-emacs-p (selected-frame) 'dummy))
	 (stack (mew-alist-search mew-window-stack key)))
    (if stack
	(set-window-configuration (car-safe stack))
      (if (and mew-emacs-p
	       (nth 1 (frame-list))
	       (mew-y-or-n-p "Delete this frame? "))
	  (delete-frame)
	(switch-to-buffer (get-buffer-create mew-window-home-buffer))))
    (setq stack (cdr-safe stack))
    (setq mew-window-stack
	  (mew-alist-add mew-window-stack key stack))
    ))

;;;
;;; Window configuration
;;;

(defun mew-window-configure (nbuf action)
;;; action : summary, message, draft or list
;;; list for action (1 0)  for Summary only
;;; list for action (3 10) for Summary and Message
  (if (equal action 'summary)
      (mew-current-set 'message nil))
  (let* ((windows
	  (if (listp action) 
	      action
	    (car (cdr (assq action mew-window-configuration)))))
	 (obufwin (get-buffer-window (current-buffer)))
	 (msgwin  (get-buffer-window (mew-buffer-message)))
	 (height nil) (winsum nil) (sumheight 0) (msgheight 0))
    (cond 
     (mew-window-use-full
      ;; Delete other windows and use full emacs window.
      (delete-other-windows)
      (setq height (window-height (selected-window))))
     (t
      (setq height
	    (+ (if obufwin (window-height obufwin) 0)
	       (if msgwin  (window-height msgwin)  0)
	       ))))
    (if (get-buffer (mew-buffer-message))
	(delete-windows-on (mew-buffer-message))
      (save-excursion
	(set-buffer (get-buffer-create (mew-buffer-message)))
	;; "truncate?" is asked in Message mode.
	;; so set the same toolbar as Sumamry mode
	(if mew-icon-p
	    (set-specifier default-toolbar
			   (cons (current-buffer) mew-summary-toolbar)))
	(mew-message-mode)))
    (setq winsum (apply (function +) windows))
    (if (not (zerop (nth 0 windows)))
	(setq sumheight (max window-min-height
			     (/ (* height (nth 0 windows)) winsum))))
    (if (and (equal action 'summary) (equal (% sumheight 2) 1)) 
	(setq sumheight (1+ sumheight)))
    (if (not (zerop (nth 1 windows)))
	(setq msgheight (max window-min-height
			     (- height sumheight))))
    (setq height (+ sumheight msgheight))
    (if (null (zerop sumheight))
	(switch-to-buffer nbuf 'norecord))
    (if (zerop msgheight)
	()
      (split-window nil sumheight)
      (other-window 1)
      (switch-to-buffer (mew-buffer-message) 'norecord))
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Folders
;;;

(defmacro mew-folder-make-alist (list)
  (` (mapcar (function mew-folder-pair) (, list))))

;; never defmacro for (mapcar)
(defun mew-folder-pair (string)
  (cons (format "+%s" string) (format "+%s" string)))

(defmacro mew-refile-make-alist (list)
  (` (mapcar (function mew-refile-pair) (, list))))
(defun mew-refile-pair (string)
  (if (mew-folders-ignore-p string)
      nil
    (cons (downcase (file-name-nondirectory (directory-file-name string)))
	  (format "+%s" (directory-file-name string)))

    ))

(defun mew-folders-ignore-p (string)
  (let ((ilist mew-folders-ignore))
    (catch 'ignore
      ;; while always returns nil
      (while ilist
	(if (string-match (concat "^" (car ilist))
			  string)
	    (throw 'ignore t))
	(setq ilist (cdr ilist))))))

(defun mew-folder-make-list ()
  (save-excursion
    (let ((case-fold-search t)
	  (folders ())
	  (folder nil)
	  (start nil))
      (set-buffer (get-buffer-create mew-buffer-tmp))
      (erase-buffer)
      (if (file-exists-p mew-folders-file)
	  (insert-file-contents mew-folders-file)
	(call-process mew-prog-folders nil t nil "-fast" "-recurse")
	)
      (goto-char (point-min))
      (while (not (eobp))
        (setq start (point))
        (forward-line)
	(setq folder (mew-buffer-substring start (1- (point))))
	(if (string-match (concat "^" (car folders) "/") folder)
	    (setq folders 
		  (cons folder 
			(cons (concat (car folders) "/"	)
			      (cdr folders))))
	  (setq folders (cons folder folders)))
	)
      folders ;; return value
      )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Bootstrap
;;;

(defvar mew-path nil)
(defvar mew-header-decode-regex nil)

(defun mew (&optional arg)
  (interactive "P")
  (mew-window-push)
  (if (null mew-path) (mew-init))
  (if arg
      (mew-summary-goto-folder) ;; C-u
    (let ((folder mew-inbox))
      (if (get-buffer folder)
	  (switch-to-buffer folder)
	(mew-summary-folder-create folder))
      (mew-summary-inc)
      )
    ))

(defun mew-send (&optional to cc subject)
  (interactive)
  (mew-current-set 'window (current-window-configuration))
  (if (null mew-path) (mew-init))
  (mew-summary-send t to cc subject)
  )

;;;
;;; Functions for boot time
;;;

(defun mew-init ()
  (mew-hello)
  (message "Setting Mew world ...")
  (mew-set-environment)
  (if mew-use-bbdb (require 'bbdb-com)) ;; bbdb is implicitly required
  (run-hooks 'mew-init-hook)
  (message "Setting Mew world ... done")
  )

(defun mew-set-environment ()
  (let ((xerr nil))
    (condition-case nil
	(progn
	  (if (string-match "^18" emacs-version)
	      (progn
		(setq xerr "Not support Emacs 18 nor Mule 1")
		(error ""))
	    )
	  (set-buffer (get-buffer-create mew-buffer-tmp))
	  (erase-buffer)
	  (if (file-exists-p mew-profile)
	      (insert-file-contents mew-profile)
	    (setq xerr (format "No %s" mew-profile))
	    (error ""))
	  (if (setq mew-path (mew-header-get-value "Path:"))
	      (setq mew-path 
		    (expand-file-name 
		     (mew-header-syntax mew-path) (expand-file-name "~")))
	    (setq xerr "No Path:")
	    (error ""))
	  (if (setq mew-draft-folder (mew-header-get-value "Draft-Folder:"))
	      (setq mew-draft-folder
		    (format "+%s" (mew-folder-to-dir
				   (mew-header-syntax mew-draft-folder))))
	    (setq xerr "No Draft-Folder:")
	    (error ""))
	  ;; mew-temp-file must be local and readable for the user only
	  ;; for privacy/speed reasons.
	  (setq mew-temp-dir (make-temp-name mew-temp-file-initial))
	  (mew-make-directory mew-temp-dir)
	  (set-file-modes mew-temp-dir 488) ;; decimal for octal 700
	  (setq mew-temp-file (expand-file-name "mew" mew-temp-dir))
	  ;;
	  (set-buffer (get-buffer-create mew-buffer-tmp))
	  (erase-buffer)
	  (call-process mew-prog-mime-encode nil t t "-v")
	  (goto-char (point-min))
	  (if (re-search-forward "version 0.10 970304" nil t)
	      ()
	    (setq xerr "mewencode version mismatch")
	    (error ""))
	  ;;
	  (erase-buffer)
	  (call-process mew-prog-msgchk nil t t "-help")
	  (goto-char (point-min))
	  (if (re-search-forward "6\\.8\\.3\\.JP" nil t)
	      ;; xxx this will be soon obsolete...
	      (setq mew-header-decode-regex 
		    "=\\?\\([^?]+\\)\\?\\(.\\)\\?\\([^?]+\\)\\?= ?")
	    (setq mew-header-decode-regex 
		  "=\\?\\([^?]+\\)\\?\\(.\\)\\?\\([^?]+\\)\\?="))
	  ;;
	  (if (null mew-alias-alist)
	      (setq mew-alias-alist (mew-alias-make-alist)))
	  ;;
	  (setq mew-folders-file 
		(expand-file-name mew-folders-file mew-path))
	  (cond 
	   ((null mew-folder-list)
	    (setq mew-folder-list (mew-folder-make-list))
	    (setq mew-folder-alist (mew-folder-make-alist mew-folder-list))
	    (setq mew-refile-alist (mew-refile-make-alist mew-folder-list))
	    )
	   )
	  (add-hook 'kill-emacs-hook (function mew-mark-process-all-folders))
	  (add-hook 'kill-emacs-hook (function mew-clean-up))
	  )
      (error (set-buffer mew-buffer-hello)
	     (insert mew-error-message)
	     (set-buffer-modified-p nil)
	     (setq buffer-read-only t)
	     (if xerr (error xerr)
	       (error "If you have ~/Mail/.mew-from-alist, remove it. Or maybe exec-path error")))
      )
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; e-mail address alias
;;;

(defun mew-alias-expand (alias)
  "The function of MH alias expansion. If ALIAS can be expanded, it 
returns the expansion. Otherwise, it returns ALIAS itself."
  (let ((expn (cdr (assoc alias mew-alias-alist))))
    ;assoc returns nil if alist is nil
    (if expn expn alias)))

(defun mew-alias-make-alist ()
  "Make alias alist with association of (alias . expantion) from
MH \"ali\" command. Currently, only \"user: user@domain\" syntax
is supported."
  (save-excursion
    (let ((case-fold-search t)
	  (alias nil)
	  (expn nil)
	  (alist nil)
	  (buf (get-buffer-create mew-buffer-tmp)))
      (set-buffer buf)
      (erase-buffer)
      (call-process mew-prog-ali nil t nil "-nolist" "-nonormalize" "-nouser")
      ;; concat separated lines by comma
      (goto-char (point-min))
      (while (re-search-forward ",$" nil t)
	(end-of-line)
	(forward-char 1)
 	(delete-backward-char 1))
      ;;
      (goto-char (point-min))
      (while (re-search-forward "^\\([^:]+\\):\\s-+\\(.*\\)$" nil t)
	(setq alias (mew-match 1)
	      expn (mew-match 2))
	;; append for first assoc comes first
	(setq alist (cons (cons alias expn) alist)))
      ;; load from-alist
      (if (not mew-from-alist)
	  (setq mew-from-alist
		(mew-alist-load mew-from-file-name)))
      (mapcar
       (function
	(lambda (arg)
	  (if (and (car arg)
		   (not (assoc (mew-header-delete-at (car arg)) alist)))
	      (setq alist 
		    (cons (cons (mew-header-delete-at (car arg)) 
				(car arg))
			  alist)))))
       mew-from-alist)
      (nreverse alist) ; return value
      )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Teer down
;;;

(defun mew-clean-up ()
  "A function to remove Mew's temporary directory recursively. 
It is typically called by kill-emacs-hook."
  (if (and mew-temp-dir (file-exists-p mew-temp-dir))
      (mew-delete-directory-recursively mew-temp-dir))
  )

(defun mew-summary-suspend ()
  "A function to suspend Mew Summary mode."
  (interactive)
  (mew-window-pop)
  ;; bury buffers
  (if window-system
      (let ((x (buffer-list))
	    (regexp (concat "^" (regexp-quote mew-buffer-message))))
	(while x
	  (if (and (buffer-name (car x))
		   (string-match regexp (buffer-name (car x))))
	      (bury-buffer (car x)))
	  (setq x (cdr x))))
    (bury-buffer (mew-buffer-message)))
  (let ((x mew-folders))
    (while x
      (if (get-buffer (car x))
	  (bury-buffer (car x)))
      (setq x (cdr x))))
  (set-buffer (car (buffer-list)))
  (run-hooks 'mew-suspend-hook)
  )

(defun mew-kill-buffer (&optional buf)
  "Common function to kill the buffer."
  (interactive)
  (if buf
      (if (get-buffer buf) (kill-buffer buf))
    (kill-buffer (current-buffer)))
  )

(defun mew-summary-quit ()
  "Exit Mew. It cleans up variables/buffers as clean as possible."
  (interactive)
  (if (null (mew-y-or-n-p "Quit Mew? "))
      ()
    (mew-window-pop)
    (setq mew-path nil)
    (setq mew-draft-folder nil)
    (setq mew-alias-alist ())
    (setq mew-folder-list ())
    (setq mew-folder-alist ())
    (setq mew-refile-alist ())
    (setq mew-cache nil)
    (if (and window-system)
	(let ((x (buffer-list))
	      (regexp (concat "^" mew-buffer-message)))
	  (while x
	    (if (and (buffer-name (car x))
		     (string-match regexp (buffer-name (car x))))
		(mew-kill-buffer (car x)))
	    (setq x (cdr x))
	    ))
      (mew-kill-buffer (mew-buffer-message)))
    (mew-kill-buffer mew-buffer-hello)
    (mew-kill-buffer mew-buffer-mime)
    (mew-kill-buffer mew-buffer-tmp)
    (let ((x (buffer-list))
	  (regex (concat "^" (regexp-quote mew-buffer-cache))))
      (while x
	(if (and (buffer-name (car x))
		 (string-match regex (buffer-name (car x))))
	    (mew-kill-buffer (car x)))
	(setq x (cdr x))
	))
    (while mew-folders
      (mew-kill-buffer (car mew-folders))
      (setq mew-folders (cdr mew-folders))
      )
    (setq mew-folders (list mew-inbox))
    (remove-hook 'kill-emacs-hook (function mew-mark-process-all-folders))
    (remove-hook 'kill-emacs-hook (function mew-clean-up))
    (mew-clean-up)
    (setq mew-temp-dir nil mew-temp-file nil)
    (run-hooks 'mew-quit-hook)
    )
  )

(defun mew-mark-process-all-folders ()
  "A function to process marked message which is not processed yet.
It is typically called by kill-emacs-hook."
  (let ((folders mew-folders))
    (save-excursion
      (while folders
	(if (bufferp (get-buffer (car folders))) ;; 
	    (progn
	      (set-buffer (car folders))
	      (if (mew-summary-mark-exist-p
		   (list mew-mark-rmm mew-mark-refile))
		  (if (mew-y-or-n-p
		       (format "Marks exists in %s. Process marks? "
			       (car folders)))
		      (mew-summary-exec)))
	      ))
	(setq folders (cdr folders))
	))
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Load Mew libraries
;;;
(require 'mew-func)
(require 'mew-minibuf)
(require 'mew-cache)
(require 'mew-mime)
(require 'mew-mark)
(require 'mew-header)
(require 'mew-pgp)
(require 'mew-encode)
(require 'mew-decode)
(require 'mew-syntax)
(require 'mew-scan)
(require 'mew-pick)
(require 'mew-summary)
(require 'mew-virtual)
(require 'mew-message)
(require 'mew-draft)
(require 'mew-attach)
(require 'mew-demo)
(require 'mew-refile)
(require 'mew-ext)
(require 'mew-fib)
;;;
;;; End of Mew
;;;
