;;;			    mew-syntax.el
;;;
;;;	      Copyright (C) 1996-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: October  2, 1996
;;; Revised: April   21, 1997
;;;

(defconst mew-syntax-version "mew-syntax.el version 0.12")

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; mew-encode-syntax
;;
;; <single>  = [ 'single file (dcr) (<pri>) (CT:)   CTE: CD: nil ]
;; <multi>   = [ 'multi  dir/ (dcr) (<pri>) ("mul") CTE: CD: nil 1*<part> ]
;; <part>    = <single> | <multi>
;; <pri>     = ("Multipart/Encrypted" "application/pgp-signature")
;;
;; mew-decode-syntax
;;
;; <message> = [ 'message hbeg hend (<pri>) ("msg") CTE: CD: CI: <part> ]
;; <single>  = [ 'single   beg  end (<pri>) (CT:)   CTE: CD: CI: ]
;; <multi>   = [ 'multi    beg  end (<pri>) ("mul") CTE: CD: CI: 1*<part> ]
;; <part>    = <message> | <single> | <multi>
;; <pri>     = results of decryption and/or verification

(defconst mew-mv: "MIME-Version:")
(defconst mew-mime-version "1.0")
(defconst mew-ct:  "Content-Type:")
(defconst mew-cte: "Content-Transfer-Encoding:")
(defconst mew-cd:  "Content-Description:")
(defconst mew-cid: "Content-ID:")

(defconst mew-mime-fields
  (list (cons mew-ct:  'analyze)
	(cons mew-cte: 'extract)
	(cons mew-cd:  'decode)
	(cons mew-cid: nil)
	))
;; See also mew-rfc822-fields

(defconst mew-syntax-magic 
  (+ (length [key beg end pri]) (length mew-mime-fields)))

(defvar mew-x-mew: "X-Mew:")

;;
;;
;;

(defmacro mew-syntax-singlepart-p (syntax)
  (` (eq (aref (, syntax) 0) 'single)))

(defmacro mew-syntax-multipart-p (syntax)
  (` (eq (aref (, syntax) 0) 'multi)))

(defmacro mew-syntax-message-p (syntax)
  (` (eq (aref (, syntax) 0) 'message)))

;;
;;
;;

(defmacro mew-syntax-get-key (syntax)
  (` (aref (, syntax) 0)))

(defmacro mew-syntax-set-key (syntax key)
  (` (aset (, syntax) 0 (, key))))

(defmacro mew-syntax-get-begin (syntax)
  (` (aref (, syntax) 1)))

(defmacro mew-syntax-set-begin (syntax begin)
  (` (aset (, syntax) 1 (, begin))))

(defmacro mew-syntax-get-end (syntax)
  (` (aref (, syntax) 2)))

(defmacro mew-syntax-set-end (syntax end)
  (` (aset (, syntax) 2 (, end))))

(defmacro mew-syntax-get-privacy (syntax)
  (` (aref (, syntax) 3)))

(defmacro mew-syntax-set-privacy (syntax privacy)
  (` (aset (, syntax) 3 (, privacy))))

(defmacro mew-syntax-get-ct (syntax)
  (` (aref (, syntax) 4)))

(defmacro mew-syntax-set-ct (syntax ct)
  (` (aset (, syntax) 4 (, ct))))

(defmacro mew-syntax-get-cte (syntax)
  (` (aref (, syntax) 5)))

(defmacro mew-syntax-set-cte (syntax cte)
  (` (aset (, syntax) 5 (, cte))))

(defmacro mew-syntax-get-cd (syntax)
  (` (aref (, syntax) 6)))

(defmacro mew-syntax-set-cd (syntax cd)
  (` (aset (, syntax) 6 (, cd))))

(defmacro mew-syntax-get-ci (syntax)
  (` (aref (, syntax) 7)))

(defmacro mew-syntax-set-ci (syntax ci)
  (` (aset (, syntax) 7 (, ci))))

(defmacro mew-syntax-get-part (syntax)
  (` (aref (, syntax) 8)))

(defmacro mew-syntax-set-part (syntax part)
  (` (aset (, syntax) 8 (, part))))

;; alias for draft syntax

(defmacro mew-syntax-get-file (syntax)
  (` (aref (, syntax) 1)))

(defmacro mew-syntax-set-file (syntax file)
  (` (aset (, syntax) 1 (, file))))

(defmacro mew-syntax-get-decrypters (syntax)
  (` (aref (, syntax) 2)))

(defmacro mew-syntax-set-decrypters (syntax decrypters)
  (` (aset (, syntax) 2 (, decrypters))))

;; for ct: parameters

(defun mew-syntax-get-member (ctl member)
  (let ((case-fold-search t)
	(regex (concat "^" member "=\"?\\([^\"]*\\)\"?$"))) ;; xxx
    (catch 'loop
      (while ctl
	(if (string-match regex (car ctl))
	    (throw 'loop (mew-match 1 (car ctl))))
	(setq ctl (cdr ctl))
	)
      )
    ))

;; need to setq
(defmacro mew-syntax-cat (syntax part)
  (` (vconcat (, syntax) (vector (, part)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Entry functions
;;

(defun mew-split-number (num-str)
  (let ((start 0)
	(end 0)
	(nums nil))
    (while (setq end (string-match "\\." num-str start))
      (setq nums (cons (string-to-int (substring num-str start end)) nums))
      (setq start (1+ end))
      )
    (nreverse (cons (string-to-int (substring num-str start nil)) nums))
    ))

(defun mew-syntax-get-entry-strnum (syntax strnum)
  (if (null strnum) 
      (mew-syntax-get-part syntax) ;; for old spec
    (mew-syntax-get-entry syntax (mew-split-number strnum)))
  )

(defun mew-syntax-get-entry (syntax nums)
  (cond
   ((null nums) syntax) ;; single & message match
   ((mew-syntax-message-p syntax)
    (mew-syntax-get-entry (mew-syntax-get-part syntax) nums))
   ((mew-syntax-multipart-p syntax)
    (if (null nums) syntax
      (mew-syntax-get-entry
	 (aref syntax (+ mew-syntax-magic (1- (car nums)))) (cdr nums))))
   ))

(defun mew-syntax-insert-entry (syntax nums entry)
  (let* ((root syntax)
	 (child entry)
	 grand parent
	 (nl (length nums))
	 (rev (reverse nums))
	 (n0 (nth 0 rev))
	 (n1 (nth 1 rev))
	 (ns (reverse (nthcdr 2 rev))))
    (cond
     ((= nl 1)
      (setq parent root)
      (mew-syntax-add-entry parent n0 child))
     (t
      (if (= nl 2)
	  (setq grand root)
	(setq grand (mew-syntax-get-entry root ns)))
      (setq parent (mew-syntax-get-entry grand (list n1)))
      (setq parent (mew-syntax-add-entry parent n0 child))
      (aset grand (+ mew-syntax-magic (1- n1)) parent)
      root
      )
     )
    ))

(defun mew-syntax-add-entry (syntax n entry)
  "Must not use in functions other than mew-syntax-insert-entry"
  (let* ((len (1+ (length syntax)))
	 (vec (make-vector len nil))
	 (cnt 0) (thr (+ mew-syntax-magic (1- n))))
    (while (< cnt thr)
      (aset vec cnt (aref syntax cnt))
      (setq cnt (1+ cnt)))
    (aset vec cnt entry)
    (setq cnt (1+ cnt))
    (while (< cnt len)
      (aset vec cnt (aref syntax (1- cnt)))
      (setq cnt (1+ cnt)))
    vec ;; return value
    ))

(defun mew-syntax-remove-entry (syntax nums)
  (let* ((root syntax)
	 grand parent
	 (nl (length nums))
	 (rev (reverse nums))
	 (n0 (nth 0 rev))
	 (n1 (nth 1 rev))
	 (ns (reverse (nthcdr 2 rev))))
    (cond
     ((= nl 1)
      (setq parent root)
      (mew-syntax-cut-entry parent n0))
     (t
      (if (= nl 2)
	  (setq grand root)
	(setq grand (mew-syntax-get-entry root ns)))
      (setq parent (mew-syntax-get-entry grand (list n1)))
      (setq parent (mew-syntax-cut-entry parent n0))
      (aset grand (+ mew-syntax-magic (1- n1)) parent)
      root
      )
     )
    ))

(defun mew-syntax-cut-entry (syntax n)
  "Must not use in functions other than mew-syntax-remove-entry"
  (let* ((len (1- (length syntax)))
	 (vec (make-vector len nil))
	 (cnt 0) (thr (+ mew-syntax-magic (1- n))))
    (while (< cnt thr)
      (aset vec cnt (aref syntax cnt))
      (setq cnt (1+ cnt)))
    (while (< cnt len)
      (aset vec cnt (aref syntax (1+ cnt)))
      (setq cnt (1+ cnt)))
    vec ;; return value
    ))

(defun mew-syntax-clear-marks (syntax nums)
  (cond
   ((null nums)
    ;; clear privacy
    (mew-syntax-set-privacy syntax nil)
    ;; get encoding back
    (mew-syntax-set-cte syntax
			(mew-syntax-get-cte
			 (mew-encode-syntax-single 
			  (mew-syntax-get-file syntax))))
    )
   (t
    (mew-syntax-clear-marks
     (aref syntax (+ mew-syntax-magic (1- (car nums)))) (cdr nums))
    )
   ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; mew-encode-syntax
;;

(defun mew-encode-syntax-single (file &optional ctl cte cd privacy decrypters)
  (let* ((attr (mew-file-attr file mew-mime-content-type))
	 (ct (mew-file-content attr)))
    (or cte (setq cte (mew-file-encoding attr)))
    (if (null ctl) (setq ctl (list ct)))
    (vconcat [single] (list file decrypters privacy ctl cte cd) [nil])
    ))

(defun mew-encode-syntax-multi (dir ct)
  (if (not (string-match "/$" dir)) (setq dir (concat dir "/")))
  (vconcat [multi] (list dir) [nil nil] (list ct) [nil nil nil])
  )

(defun mew-encode-syntax-initial (dir)
  (vconcat
   (mew-encode-syntax-multi dir mew-type-mlm)
   (list (mew-encode-syntax-single mew-draft-coverpage (list mew-ct-txt))))
  ;; ensure to guess charset ....
  )

(defun mew-encode-syntax-initial-multi (dir n)
  (let ((i 1) (ret))
    (while (<= i n)
      (setq ret (vconcat ret
			 (list (mew-encode-syntax-single (int-to-string i)))))
      (setq i (1+ i)))
    (vconcat (mew-encode-syntax-multi dir mew-type-mlm)
	     (list (mew-encode-syntax-single mew-draft-coverpage 
					     (list mew-ct-txt)))
	     ret)
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; mew-encode-syntax printer
;;

(defun mew-encode-syntax-print (syntax)
  (interactive)
  (let ((end nil)
	(nums (mew-attach-nums)))
    (cond
     ((mew-attach-p)
      (goto-char (point-max))
      (re-search-backward (concat "^" mew-draft-attach-boundary-end "$"))
      (setq end (point))
      (re-search-backward (concat "^" mew-draft-attach-boundary-begin "$"))   
      (forward-line)
      (delete-region (point) end)
      (save-excursion
	(mew-encode-syntax-print-loop syntax "" 0)
	)
      (mew-attach-goto-number 'here nums)
      (if mew-icon-p
	  (let ((toolbar))
	    (setq mew-draft-icon-spec nil)
	    (mew-encode-syntax-iconify syntax "" 0)
	    (cond
	     ((eq mew-multipart-icon-position 'left)
	      (setq toolbar (append (nreverse mew-draft-icon-spec)
				    mew-icon-separate-spec
				    mew-draft-toolbar)))
	     ((eq mew-multipart-icon-position 'right)
	      (setq toolbar (append mew-draft-toolbar
				    mew-icon-separate-spec
				    (nreverse mew-draft-icon-spec))))
	     (t (setq toolbar mew-summary-toolbar))
	     )
	    (set-specifier default-toolbar (cons (current-buffer) toolbar))
	    ))
      ))
    ))

(defun mew-encode-syntax-print-loop (syntax number level)
  (let ((buffer-read-only nil))
    (cond
     ((mew-syntax-singlepart-p syntax)
      (insert (mew-encode-syntax-format syntax number level)))
     ((mew-syntax-multipart-p syntax)
      (insert (mew-encode-syntax-format syntax number level))
      (let ((cnt mew-syntax-magic)
	    (num 1)
	    (len (length syntax))
	    (nlevel (1+ level)))
	(cond
	 ((= level 0)
	  (while (< cnt len)
	    (mew-encode-syntax-print-loop (aref syntax cnt)
					  (int-to-string num) nlevel)
	  (setq cnt (1+ cnt))
	  (setq num (1+ num)))
	  (insert (mew-encode-syntax-format [nil "." nil nil ("") nil nil nil]
					    (int-to-string num) nlevel)))
	 (t
	  (while (< cnt len)
	    (mew-encode-syntax-print-loop 
	     (aref syntax cnt)
	     (concat number "." (int-to-string num))
	     nlevel)
	  (setq cnt (1+ cnt))
	  (setq num (1+ num)))
	  (insert (mew-encode-syntax-format
		   [nil "." nil nil ("") nil nil nil]
		   (concat number "." (int-to-string num))
		   nlevel)))
	 )
	))
     )))

;012345678901234567890123456789012345678901234567890123456789012345678901234567
;<5 >nsss<16            >ss<27-2                   >s<24+2                    >
;----------- attachments -----------
;       1/                Multipart/Mixed             
;     1   CoverPage         Text/Plain(us-ascii)      
;     2   12345678901234..  Text/Plain(iso-2022-jp)   "1234567890123456789012.."
;     3   .                                           
;--------0-1-2-3-4-5-6-7-8-9--------

(defun mew-encode-syntax-format (syntax number level)
  ;; level starts 1
  (let* ((file (mew-syntax-get-file syntax))
	 (ctl (mew-syntax-get-ct syntax))
	 (ct (capitalize (car ctl)))
	 (char (mew-syntax-get-member ctl "charset"))
	 (cte (mew-syntax-get-cte syntax))
	 (cd (mew-syntax-get-cd syntax))
	 (decrypters (mew-syntax-get-decrypters syntax))
	 (privacy (mew-syntax-get-privacy syntax)) ctm ctp
	 (space " ") (space-char 32)
	 (quote "\"") 
	 (cnt "..") (lcnt (length cnt))
	 (marks "") (lm 5)
	 (lf 16) (lfc (- lf lcnt))
	 (lt 27) (ltc (- lt lcnt))
	 (ld 24) (ldc (- ld lcnt))
	 (case-fold-search t))

    (if (string= (capitalize mew-ct-txt) ct)
	(if char
	    (setq ct (concat ct "(" char ")"))
	  (setq ct (concat ct "(guess)"))))

    (if (null privacy)
	(if (null cte)
	    (setq marks "     ")
	  (cond
	   ((equal cte mew-b64) (setq marks "B    "))
	   ((equal cte mew-qp)  (setq marks "Q    "))
	   ((equal cte mew-xg)  (setq marks "G    "))
	   ))
      (while privacy
	(setq ctm (nth 0 (car privacy)))
	(setq ctp (nth 1 (car privacy)))
	(setq privacy (cdr privacy))
	(cond
	 ((string-match "pgp"  ctp) (setq marks (concat marks "P"))) 
	 ((string-match "moss" ctp) (setq marks (concat marks "M"))) 
	 )
	(cond
	 ((string-match mew-ct-mle ctm) (setq marks (concat marks "E"))) 
	 ((string-match mew-ct-mls ctm) (setq marks (concat marks "S"))) 
	 )
	)
      (setq marks (concat marks "     "))
      )
    (setq marks (substring marks 0 lm))

    (concat
     marks
     number
     (if (not (equal number "")) space)
     space space

     (if (< lf (length file))
	 (concat (substring file 0 lfc) cnt)
       (concat file (make-string (- lf (length file)) space-char)))
     space space

     (let* ((tmp (* 2 level))
	    (vlt (- lt tmp))
	    (vltc (- ltc tmp))
	    (lct (length ct)))
       (if (< vlt lct)
	   (concat (substring ct 0 vltc) cnt)
	 (concat ct (make-string (- vlt lct) space-char))))
     space

     (let ((cd-or-dec cd))
       (if decrypters (setq cd-or-dec decrypters))
       (if (null cd-or-dec)
	   ""
	 (if (< ld (length cd-or-dec))
	     (concat quote (substring cd-or-dec 0 ldc) cnt quote)
	   (concat quote cd-or-dec quote))))
     "\n"
     )
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; mew-encode-syntax iconifyer
;;

(setq mew-draft-icon-spec nil)

(defun mew-encode-syntax-iconify (syntax number level)
  (let ((buffer-read-only nil)
	(ct (car (mew-syntax-get-ct syntax)))
	(cd (or (mew-syntax-get-cd syntax) "")))
    (cond
     ((mew-syntax-singlepart-p syntax)
      (setq mew-draft-icon-spec
	    (cons
	     (vector
	      (nth 4 (mew-assoc-match2 ct mew-mime-content-type 0))
	      'mew-draft-button
	      t
	      (format "<%s> (%s) %s" number ct cd))
	     mew-draft-icon-spec)))
     ((mew-syntax-multipart-p syntax)
      (setq mew-draft-icon-spec
	    (cons
	     (vector
	      (nth 4 (mew-assoc-match2 ct mew-mime-content-type 0))
	      'mew-draft-button
	      t
	      (format "<%s> (%s) %s" number ct cd))
	     mew-draft-icon-spec))
      (let ((cnt mew-syntax-magic)
	    (num 1)
	    (len (length syntax))
	    (nlevel (1+ level)))
	(cond
	 ((= level 0)
	  (while (< cnt len)
	    (mew-encode-syntax-iconify (aref syntax cnt)
				       (int-to-string num) nlevel)
	    (setq cnt (1+ cnt))
	    (setq num (1+ num)))
	  (setq mew-draft-icon-spec
		(cons
		 (vector
		  mew-icon-blank
		  'mew-draft-button
		  t
		  (format "<%s> (Attach Here)" (int-to-string num)))
		 mew-draft-icon-spec))
	  )
	 (t
	  (while (< cnt len)
	    (mew-encode-syntax-iconify
	     (aref syntax cnt)
	     (concat number "." (int-to-string num))
	     nlevel)
	    (setq cnt (1+ cnt))
	    (setq num (1+ num)))
	  (setq mew-draft-icon-spec
		(cons
		 (vector
		  mew-icon-blank
		  'mew-draft-button
		  t
		  (format "<%s> (Attach Here)" 
			  (concat number "." (int-to-string num))))
		 mew-draft-icon-spec))
	  )
	 ))
      ))
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; mew-decode-syntax
;;

(defun mew-decode-syntax-rfc822 (&optional msg-head)
  ;; msg-head may include CD:
  (if (null msg-head) (setq msg-head (mew-decode-syntax-rfc822-head)))
  (vconcat msg-head (vector (mew-decode-syntax-text))))

(defun mew-decode-syntax-rfc822-head ()
  ;; hend may be 0 since (point-min) is 1
  (vector 'message (point-min) (1- (point)) nil mew-type-msg nil nil nil))

(defun mew-decode-syntax-text ()
  (vector 'single (point) (point-max) nil mew-type-txt nil nil nil))

(defconst mew-encode-syntax-multi-head 
  (vector 'multi nil nil nil mew-type-mlm nil nil nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; mew-decode-syntax printer
;;

(setq mew-decode-result nil)

(defun mew-decode-syntax-print (msg syntax buf)
  (let ((cbuf (current-buffer)))
    (set-buffer buf)
    (forward-line)
    (set-marker mew-current-marker (point) (current-buffer))
    (let* ((buffer-read-only nil)
	   (multi (mew-syntax-get-part syntax))
	   (privacy (mew-syntax-get-privacy multi))
	   (pref "\t")
	   (cnt mew-syntax-magic)
	   (num 1)
	   (len (length multi)))
      (if privacy
	  (setq mew-decode-result (concat mew-x-mew: " <all> " privacy "\n")))
      (while (< cnt len)
	(mew-decode-syntax-print-loop 
	 (aref multi cnt)
	 (concat pref (int-to-string num)))
	(setq cnt (1+ cnt))
	(setq num (1+ num))
	)
      )
    (goto-char (marker-position mew-current-marker))
    (forward-line -1)
    (set-buffer-modified-p nil)
    (if mew-icon-p
	(let ((toolbar))
	  (setq mew-summary-icon-spec nil)
	  (mew-decode-syntax-iconify msg syntax)
	  (cond
	   ((eq mew-multipart-icon-position 'left)
	    (setq toolbar (append (nreverse mew-summary-icon-spec)
				  mew-icon-separate-spec
				  mew-summary-toolbar)))
	   ((eq mew-multipart-icon-position 'right)
	    (setq toolbar (append mew-summary-toolbar
				  mew-icon-separate-spec
				  (nreverse mew-summary-icon-spec))))
	   (t (setq toolbar mew-summary-toolbar))
	   )
	  (set-specifier default-toolbar (cons (current-buffer) toolbar))
	  )
      )
    (set-buffer cbuf)
    ))

(defun mew-decode-syntax-print-loop (syntax pref)
  (let ((po (point))
	(ct (capitalize (car (mew-syntax-get-ct syntax))))
	(cd (or (mew-syntax-get-cd syntax) ""))
	(privacy (mew-syntax-get-privacy syntax)))
    (if privacy
	(setq mew-decode-result
	      (concat mew-decode-result
		      (format (concat mew-x-mew: " <%s> ")
			      (progn (string-match "[1-9.]+" pref)
				     (mew-match 0 pref)))
		      privacy
		      "\n")))
    (cond
     ((mew-syntax-message-p syntax)
      (insert (format "%s\t%s\t%s\n" pref ct cd))
      (mew-summary-highlight-lines-region po (point) t)
      (if (mew-syntax-multipart-p (mew-syntax-get-part syntax))
	  (mew-decode-syntax-print-loop
	   (mew-syntax-get-part syntax)
	   (concat "\t" pref))))
     ((mew-syntax-singlepart-p syntax)
      (insert (format "%s\t%s\t%s\n" pref ct cd))
      (mew-summary-highlight-lines-region po (point) t))
     ((mew-syntax-multipart-p syntax)
      (let ((cnt mew-syntax-magic)
	    (num 1)
	    (len (length syntax)))
	(while (< cnt len)
	  (mew-decode-syntax-print-loop 
	   (aref syntax cnt)
	   (concat pref "." (int-to-string num)))
	  (setq cnt (1+ cnt))
	  (setq num (1+ num))
	  )))
     )
    ))

(defun mew-decode-syntax-delete ()
  (let ((buf (current-buffer)))
    (if (null (marker-position mew-current-marker))
	()
      (set-buffer (marker-buffer mew-current-marker))
      (if (null mew-icon-p)
	  ()
	(setq mew-summary-icon-spec nil)
	(set-specifier default-toolbar
		       (cons (current-buffer) mew-summary-toolbar))
	)
      (set-marker mew-current-marker2 (point))
      (goto-char (marker-position mew-current-marker))
      (let ((start (point))
	    (buffer-read-only nil))
	(while (looking-at "^\t")
	  (forward-line))
	(delete-region start (point)))
      (set-marker mew-current-marker nil)
      (goto-char (marker-position mew-current-marker2))
      (set-buffer buf)
      ))
  )

(defun mew-decode-syntax-set-privacy (syntax)
  (let ((privacy (mew-syntax-get-privacy (mew-syntax-get-part syntax))))
    (if privacy
	(setq mew-decode-result (concat mew-x-mew: " <all> " privacy "\n")))
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; mew-decode-syntax iconifyer
;;

(setq mew-summary-icon-spec nil)

(defun mew-decode-syntax-iconify (msg syntax)
  ;; syntax is message containing multipart.
  (let* ((multi (mew-syntax-get-part syntax))
	 (cnt mew-syntax-magic)
	 (num 1)
	 (len (length multi)))
    (while (< cnt len)
      (mew-decode-syntax-iconify-loop 
       msg
       (aref multi cnt)
       (int-to-string num))
      (setq cnt (1+ cnt))
      (setq num (1+ num))
      )
    ))

(defun mew-decode-syntax-iconify-loop (msg syntax pref)
  (let ((ct (capitalize (car (mew-syntax-get-ct syntax))))
	(cd (or (mew-syntax-get-cd syntax) "")))
    (cond
     ((mew-syntax-message-p syntax)
      (setq mew-summary-icon-spec 
	    (cons
	     (vector
	      (nth 4 (mew-assoc-match2 ct mew-mime-content-type 0))
	      'mew-summary-button
	      t
	      (format "%s <%s> (%s) %s" msg pref ct cd))
	     mew-summary-icon-spec))
      (if (mew-syntax-multipart-p (mew-syntax-get-part syntax))
	  (mew-decode-syntax-iconify-loop
	   msg
	   (mew-syntax-get-part syntax)
	   pref)))
     ((mew-syntax-singlepart-p syntax)
      (setq mew-summary-icon-spec 
	    (cons
	     (vector
	      (nth 4 (mew-assoc-match2 ct mew-mime-content-type 0))
	      'mew-summary-button
	      t
	      (format "%s <%s> (%s) %s" msg pref ct cd))
	     mew-summary-icon-spec))
      )
     ((mew-syntax-multipart-p syntax)
      (let ((cnt mew-syntax-magic)
	    (num 1)
	    (len (length syntax)))
	(while (< cnt len)
	  (mew-decode-syntax-iconify-loop
	   msg
	   (aref syntax cnt)
	   (concat pref "." (int-to-string num)))
	  (setq cnt (1+ cnt))
	  (setq num (1+ num))
	  )))
     )
    ))

(provide 'mew-syntax)
