;;;			    mew-summary.el
;;;
;;;	      Copyright (C) 1996-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: October  2, 1996
;;; Revised: April   21, 1997
;;;

(defconst mew-summary-version "mew-summary.el version 0.17")

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; User customize variables
;;;

(defvar mew-summary-mode-map    nil)

(defvar mew-summary-mode-menu-spec
  '("Mew"
    ["Show" mew-summary-show t]
    ["Next part" mew-summary-display-down t]
    ["Previous part" mew-summary-display-up t]
    ["Top" mew-summary-display-top t]
    ["Bottom" mew-summary-display-bottom t]
    ["Jump" mew-summary-jump-message t]
    "----"
    ["Delete" mew-summary-rmm (equal major-mode 'mew-summary-mode)]
    ["Refile" mew-summary-refile (equal major-mode 'mew-summary-mode)]
    ["Mark mark" mew-summary-mark-mark t]
    ["Mark hop" mew-summary-mark-hop t]
    ["Undo" mew-summary-undo t]
    ["Undo all" mew-summary-undo-all t]
    ["Execute" mew-summary-exec (equal major-mode 'mew-summary-mode)]
    ["Suspend" mew-summary-suspend t]
    ["Quit" mew-summary-quit t]
    "----"
    ("Manipulate folder"
     ["Inc" mew-summary-inc t]
     ["Scan" mew-summary-scan (equal major-mode 'mew-summary-mode)]
     ["Pack" mew-summary-pack (equal major-mode 'mew-summary-mode)]
     ["Sort" mew-summary-sort (equal major-mode 'mew-summary-mode)]
     ["Burst" mew-summary-burst t]
     ["Go to folder" mew-summary-goto-folder t]
     )
    ("Manipulate file"
     ["Save" mew-summary-save t]
     ["Convert to local charset" mew-summary-convert-local-charset t]
     ["Display X-Face" mew-summary-x-face t]
     )
    ("Write/Reply/Forward"
     ["Write a mail" mew-summary-send t]
     ["Reedit" mew-summary-reedit t]
     ["Reply" mew-summary-reply t]
     ["Reply with citation" mew-summary-reply-with-citation t]
     ["Forward" mew-summary-forward t]
     ["Multi forward" mew-summary-multi-forward t]
     )
    ("Select"
     ["Search then Mark" mew-summary-search-mark
      (equal major-mode 'mew-summary-mode)]
     ["Search" mew-summary-search (equal major-mode 'mew-summary-mode)]
     ["Virtual mode" mew-summary-virtual (equal major-mode 'mew-summary-mode)]
     )
    ("Misc"
     ["Recenter" mew-summary-recenter t]
     ["Uudecode" mew-summary-uudecode t]
     ["Unshar" mew-summary-unshar t]
     ["Print" mew-summary-print t]
     ["Pipe message" mew-summary-pipe-message t]
     ["Isearch forward" mew-summary-isearch-forward t]
     ["Isearch backward" mew-summary-isearch-backward t]
     ["Toggle disp msg" mew-summary-toggle-disp-msg t]
     ["Toggle analysis" mew-summary-toggle-analysis t]
     ["Kill Sub-Process"
      mew-summary-kill-subprocess 
      (and (processp mew-summary-buffer-process)
	   (equal major-mode 'mew-summary-mode))]
     )
    )
  )

(if mew-summary-mode-map
    ()
  (setq mew-summary-mode-map (make-sparse-keymap))
  (define-key mew-summary-mode-map " "    'mew-summary-show)
  (define-key mew-summary-mode-map "."    'mew-summary-display)
  (define-key mew-summary-mode-map "<"    'mew-summary-display-top)
  (define-key mew-summary-mode-map ">"    'mew-summary-display-bottom)
  (define-key mew-summary-mode-map "\177" 'mew-summary-prev-page)
  (define-key mew-summary-mode-map "\r"   'mew-summary-scroll-up)
  (define-key mew-summary-mode-map "\e\r" 'mew-summary-scroll-down)
  (define-key mew-summary-mode-map "g"    'mew-summary-goto-folder)
  (define-key mew-summary-mode-map "j"    'mew-summary-jump-message)
  (define-key mew-summary-mode-map "i"    'mew-summary-inc)
  (define-key mew-summary-mode-map "a"    'mew-summary-reply)
  (define-key mew-summary-mode-map "A"    'mew-summary-reply-with-citation)
  (define-key mew-summary-mode-map "E"    'mew-summary-reedit)
  (define-key mew-summary-mode-map "\ee"  'mew-summary-edit-again)
  (define-key mew-summary-mode-map "f"    'mew-summary-forward)
  (define-key mew-summary-mode-map "F"    'mew-summary-multi-forward)
  (define-key mew-summary-mode-map "r"    'mew-summary-redist)
  (define-key mew-summary-mode-map "@"    'mew-summary-mark-mark)
  (define-key mew-summary-mode-map "*"    'mew-summary-mark-hop)
  (define-key mew-summary-mode-map "y"    'mew-summary-save)
  (define-key mew-summary-mode-map "u"    'mew-summary-undo)
  (define-key mew-summary-mode-map "U"    'mew-summary-undo-all)
  (define-key mew-summary-mode-map "n"    'mew-summary-display-down)
  (define-key mew-summary-mode-map "p"    'mew-summary-display-up)
  (define-key mew-summary-mode-map "N"    'mew-summary-display-hop-down)
  (define-key mew-summary-mode-map "P"    'mew-summary-display-hop-up)
  (define-key mew-summary-mode-map "w"    'mew-summary-send)
  (define-key mew-summary-mode-map "B"    'mew-summary-burst)
  (define-key mew-summary-mode-map "Z"    'mew-status-update)
  (define-key mew-summary-mode-map "!"    'mew-summary-refile-again)
  (define-key mew-summary-mode-map "#"    'mew-summary-print)
  (define-key mew-summary-mode-map "|"    'mew-summary-pipe-message)
  (define-key mew-summary-mode-map "q"    'mew-summary-suspend)
  (define-key mew-summary-mode-map "Q"    'mew-summary-quit)
  (define-key mew-summary-mode-map "\C-c\C-e" 'mew-summary-execute-external)
  (define-key mew-summary-mode-map "\C-c\C-s" 'mew-summary-isearch-forward)
  (define-key mew-summary-mode-map "\C-c\C-r" 'mew-summary-isearch-backward)
  (define-key mew-summary-mode-map "\C-c\C-o" 
    'mew-summary-jump-to-draft-buffer)
  (define-key mew-summary-mode-map "\el"  'mew-summary-recenter)
  (define-key mew-summary-mode-map "\et"  'mew-summary-uudecode)
  (define-key mew-summary-mode-map "\C-c\C-l"
    'mew-summary-convert-local-charset)
  (define-key mew-summary-mode-map "\es"  'mew-summary-unshar)
  (define-key mew-summary-mode-map "v"    'mew-summary-toggle-disp-msg)
  (define-key mew-summary-mode-map "\ea"  'mew-summary-toggle-analysis)
  (define-key mew-summary-mode-map "\C-c\C-x" 'mew-summary-x-face)
  (define-key mew-summary-mode-map "\C-c\C-q" 'mew-kill-buffer)
  ;;
  ;; not provided in virtual mode
  ;;
  (define-key mew-summary-mode-map "o"    'mew-summary-refile)
  (define-key mew-summary-mode-map "O"    'mew-summary-pack)
  (define-key mew-summary-mode-map "s"    'mew-summary-scan)
  (define-key mew-summary-mode-map "S"    'mew-summary-sort)
  (define-key mew-summary-mode-map "d"    'mew-summary-rmm)
  (define-key mew-summary-mode-map "x"    'mew-summary-exec)
  (define-key mew-summary-mode-map "X"    'mew-summary-exec-current)
  (define-key mew-summary-mode-map "V"    'mew-summary-virtual)
  (define-key mew-summary-mode-map "/"    'mew-summary-search)
  (define-key mew-summary-mode-map "?"    'mew-summary-search-mark)
  (define-key mew-summary-mode-map "m"	  (make-sparse-keymap))
  (define-key mew-summary-mode-map "mo"   'mew-summary-mark-refile)
  (define-key mew-summary-mode-map "md"   'mew-summary-mark-rmm)
  (define-key mew-summary-mode-map "m@"   'mew-summary-mark-at)
  (define-key mew-summary-mode-map "m*"   'mew-summary-mark-asterisk)
  (define-key mew-summary-mode-map "ms"   'mew-summary-swap-mark<->hop)
  (define-key mew-summary-mode-map "mr"   'mew-summary-mark-regexp)
  (define-key mew-summary-mode-map "ma"   'mew-summary-mark-all)
  (define-key mew-summary-mode-map "mu"   'mew-summary-mark-undo-all)
  (define-key mew-summary-mode-map "\C-c\C-k" 'mew-summary-kill-subprocess)
  ;;
  (if mew-xemacs-p
      (define-key mew-summary-mode-map 'button2 'mew-summary-mouse-show)
    (define-key mew-summary-mode-map [mouse-2] 'mew-summary-mouse-show)
    (easy-menu-define
     mew-summary-mode-menu
     mew-summary-mode-map
     "Menu used in Mew summary mode."
     mew-summary-mode-menu-spec))
  )

(defvar mew-summary-mode-toolbar-menu
  '("Mew Part Commands"
    ["Save" mew-summary-save t]
    ["Reply" mew-summary-reply t]
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 
;;;

(defvar mew-current-marker (make-marker))
(defvar mew-current-marker2 (make-marker))

(defvar mew-field-delete
  '("Date:" "Message-Id:" "Return-Path:" "Received:"
    "From:" "Sender:" "Delivery-Date:" "X-Mailer:"))

(defvar mew-field-resent
  '("Resent-From:" "Resent-Sender:" "Resent-To:"
    "Resent-Cc:" "Resent-Date:" "Resent-Message-Id:" "Resent-Reply-To:"))

(defvar mew-last-shell-command "")

(defvar mew-trash-folder nil
  "*If non-nil, messages to be deleted are refiled to the defined folder")

(defvar mew-summary-message-regex "^ *\\([0-9]+\\)")

(defvar mew-summary-part-regex "^\t+\\([0-9.]+\\)")

(defvar mew-summary-edit-again-regex
  "----- Original message follows -----\\|----- Unsent message follows -----")

(defvar mew-redist-needs-full-contents nil
  "If non-nil, MH post is supposed to spawn sendmail and pass a message
with PIPE. If nil, it is supposed to speak SMTP.")

(defvar mew-redist-format1 "mhdist=1 %s %s"
  "For MH/post that spawns sendmail")
(defvar mew-redist-format2 "mhdist=1 mhaltmsg=%s %s %s"
  "For MH/post that speaks SMTP")


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Summary mode
;;;

(defun mew-summary-mode ()
  "Major mode for reading messages.
The keys that are defined for this mode are:

SPC	Read through messages. See mew-summary-show-direction to set 
	'up, 'down, 'next(current direction) or 'stop. Default is 'down.
DEL	Back scroll this message.
	Unnecessary header fields are hidden over the window. Type DEL
	to see them when message are displayed.
.	Display this message or part. 
	If without MIME analysis, force to analyze this message.

RET	1 line scroll up this message.
ESC RET 1 line scroll down this message.

C-n	Go to the next line.
C-p	Go to the previous line.
n	Display below message or part.
p	Display above message or part.
N	Jump to below * marked message or display below message
	around multipart.
P	Jump to above * marked message or display above message
	around multipart.
j	Jump to a message according to inputed number.

i	Incorporate +inbox asynchronously.
s	Scan this folder asynchronously. 
g	Go to inputed folder.

w	Prepare draft to send a message.
a	Reply to this message or this part.
A	Reply to this message and insert it in draft buffer.
f	Forward this message as MIME format.
F	Forward @ marked messages as MIME format.
E	Edit this message again to send. Or edit this rfc822 part
	typically included MIME-encapsulated error message.
	In a draft folder, it just edits the message. Otherwise, 
	copy the message to draft folder, then edit.
ESC e	Edit an old fashioned error mail in which the original message 
	is encapsulated after \"----- Original message follows -----\".
r	Re-distribute this message with Resent-To:. It is strongly 
	discouraged to use this command since beginners are always 
	confused. Please use \"f\" instead.

v	Toggle summary window only and summary & message windows.
	If you select summary window only, \"D\" never displays the 
	next message. So, you can set D marks quickly.
ESC a	Toggle with/out MIME analysis.
ESC l	Recenter this folder.

o	Mark this message or this part's message with refile
	mark(default is \"o\"). If already marked with \"o\", it
	prints to which folder this message is refiled.
	This can overlay other marks. When it overlays, the cursor stays
	on the message. If it marks newly, displays the next message.
!	Refile this message or part to the previous refile folder.
d	Mark this message with delete mark(default is \"D\").
	This can overlay other marks. When it overlays, the cursor stays
	on the message. If it marks newly, displays the next message.
*	Mark this message with \"*\". Use N or P to jump to * marked message.
	It can overlay \"@\". The cursor stays always.
	See also 'mo', 'md', 'mr', and 'ma'.
@	Mark this message with \"@\" for F, ESC s, and ESC t.
	It can overlay \"*\". The cursor stays always.
u	Cancel mark on this message.
U	Cancel all inputed mark.
x	Process marked messages. To cancel \"*\" mark, use u or U.

mo	Mark all messages whose marks are \"*\" with refile mark(\"o\").
	This is very convenient to refile all messages picked by '?'.
md	Mark all messages whose marks are \"*\" with delete mark(\"D\").
mr	Mark all messages matched with inputed regular expression with \"*\".
ma	Mark all messages with \"*\" which are not marked.
m*	Change mark \"@\" into mark \"*\".
m@	Change mark \"*\" into mark \"@\".
ms	Swap mark \"@\" and \"*\".

C-cC-s	Incremental search forward on the message buffer.
C-cC-r	Incremental search backward on the message buffer.

ESC s	Apply unshar on @ marked messages.
ESC t	Apply uudecode on @ marked messages.

/	Pick messages according to inputed condition, then scan.
?	Pick messages according to inputed condition, then mark *.
V	Goto virtual mode which gives a single view to picked messages
	from multiple folders. Enter virtual folder name, comma-separated
	folders, and pick pattern.

y	Copy this message or save this part as inputed file name.
#	Print this message or this part.
|	Pipe this message.

S	Sort messages in this folder.
O	Pack messages in this folder.

q	Switch to other buffer.
Q	Exit mew.
C-cC-q	Kill this buffer.

C-cC-l	Localize the transformed charset.
C-cC-x	Display xface.

Z	Update aliases and folders list. If you create a new alias or folder
	with native MH, try this.

C-cC-k	Kill a process in Summary mode such as 'inc' and 'scan'.
	Sometime a process accidentally remains in Summary mode. 
	In this situation, you cannot execute 's', 'i', nor 'x'.
	Use this command to solve this problem.

Range means as follows;
	all, update, 
	<num1>-<num2>, <num>:+N, <num>:-N, 
	first:N, prev:N, next:N, last:N

Use 'all' to flush the summary buffer.  'update' means the range
between the last message included in summary mode + 1 and the real last
message on the folder.

Pick condition means as follews;
	<cond1> -and <cond2>, <cond1> -or <cond2>
	-from <pat>, -to <pat>, -search <pat> 
	-date <date>, -before <date>, -after <date>
	--xxx <pat>
	-lbrace or \"(\", -rbrace or \")\"
"
  (interactive)
  (setq major-mode 'mew-summary-mode)
  (setq mode-name "Summary")
  (use-local-map mew-summary-mode-map)
  (setq buffer-read-only t)
  (setq truncate-lines t)
  ;; xxx gee dirty
  (if (equal (nth (- (length mode-line-format) 2) mode-line-format)
	     '(-3 . "%p"))
      (setq mode-line-format
	    (let ((mlf (copy-sequence mode-line-format))
		  (l (length mode-line-format)))
	      (setcdr (nthcdr (- l 3) mlf)
		      '("[" mew-summary-buffer-left-msgs " more]" "-%-"))
	      mlf)
	    ))
  (if mew-xemacs-p
      (progn
	(if mew-icon-p
	    (set-specifier default-toolbar
			   (cons (current-buffer) mew-summary-toolbar)))
        (set-buffer-menubar current-menubar)
        (add-submenu nil mew-summary-mode-menu-spec)
	))
  (if mew-highlight-lines-use
      (if mew-xemacs-p
	  (setq mode-motion-hook 'mode-motion-highlight-line)
	(if (save-excursion 
	      (goto-char (point-min))
	      (not (overlays-at (point))))
	    (mew-summary-highlight-lines-region (point-min) (point-max)))))
  (run-hooks 'mew-summary-mode-hook)
  )

(defun mew-summary-folder-name ()
  (cond 
   ((equal major-mode 'mew-summary-mode)
    (buffer-name))
   ((equal major-mode 'mew-virtual-mode)
    (save-excursion
      (beginning-of-line)
      (if (looking-at ".*\r \\(\\+.*\\) \\(.*\\)$")
	  (mew-match 1)
	nil
	)))
   (t nil)
   )
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; get number
;;;
    
(defun mew-summary-message-number ()
  (cond 
   ((equal major-mode 'mew-summary-mode)
    (let ((event last-command-event)
	  ret str)
      (if (and mew-icon-p	
	       ;; exclude press button 2 in summary buffer.
	       ;; exclude pulldown menu in Summary mode.
	       ;; exclude popup menu of multipart icon because
	       ;; the cursor has already moved.
	       (or (button-press-event-p event)     ;; right button
		   (button-release-event-p event))) ;; left button
	  (progn
	    (setq str (toolbar-button-help-string last-pressed-toolbar-button))
	    (if (string-match "^\\([0-9]+\\) " str)
		(setq ret (mew-match 1 str)))
	    ))
      (if ret
	  ret
	(save-excursion
	  (beginning-of-line)
	  (if (looking-at mew-summary-message-regex)
	      (mew-match 1)
	    nil)
	  ))
      ))
   ((equal major-mode 'mew-virtual-mode)
    (save-excursion
      (beginning-of-line)
      (if (looking-at ".*\r \\(\\+.*\\) \\(.*\\)$")
	  (mew-match 2)
	nil)))
   (t nil)
   ))

(defun mew-summary-part-number ()
  (let ((event last-command-event)
	ret str)
    (if (and mew-icon-p	
	     ;; exclude press button 2 in summary buffer.
	     ;; exclude pulldown menu in Summary mode.
	     ;; exclude popup menu of multipart icon because
	     ;; the cursor has already moved.
	     (or (button-press-event-p event)     ;; right button
		 (button-release-event-p event))) ;; left button
	(progn
	  (setq str (toolbar-button-help-string last-pressed-toolbar-button))
	  (if (string-match "^[0-9]+ <\\([0-9.]+\\)> " str)
	      (setq ret (mew-match 1 str)))
	  ))
    (if ret
	ret
      (save-excursion
	(beginning-of-line)
	(if (looking-at mew-summary-part-regex)
	    (mew-match 1)
	  nil)
	))
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Folder
;;;

(defun mew-summary-goto-folder ()
  (interactive)
  (let* ((folder (mew-input-folder mew-inbox))
	 (dir (mew-expand-file-name folder)))
    (if (null (file-directory-p dir))
	(message "No such folder %s" folder)
      (if (get-buffer folder)
	  (switch-to-buffer folder)
	(mew-summary-folder-create folder))
      (mew-summary-scan)
      )))

(defun mew-summary-folder-create (folder)
  (let ((cache (mew-expand-file-name mew-summary-cache-file folder)))
    (switch-to-buffer (get-buffer-create folder))
    (cond
     ((file-exists-p cache)
      (insert-file-contents cache)
      (setq mew-summary-buffer-folder-cache-time 
	    (nth 5 (file-attributes cache))))
     )
    (mew-summary-mode)
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Summary file cache
;;;

;; xxx will be deleted
(defun mew-file-chase-links (filename)
  "Chase links in FILENAME until a name that is not a link.
Does not examine containing directories for links,
unlike `file-truename'."
  (let (tem (count 100) (newname filename))
    (while (setq tem (file-symlink-p newname))
      (if (= count 0)
	  (error "Apparent cycle of symbolic links for %s" filename))
      ;; In the context of a link, `//' doesn't mean what Emacs thinks.
      (while (string-match "//+" tem)
	(setq tem (concat (substring tem 0 (1+ (match-beginning 0)))
			  (substring tem (match-end 0)))))
      ;; Handle `..' by hand, since it needs to work in the
      ;; target of any directory symlink.
      ;; This code is not quite complete; it does not handle
      ;; embedded .. in some cases such as ./../foo and foo/bar/../../../lose.
      (while (string-match "\\`\\.\\./" tem)
	(setq tem (substring tem 3))
	(setq newname (file-name-as-directory
		       ;; Do the .. by hand.
		       (directory-file-name
			(file-name-directory
			 ;; Chase links in the default dir of the symlink.
			 (mew-file-chase-links
			  (directory-file-name
			   (file-name-directory newname))))))))
      (setq newname (expand-file-name tem (file-name-directory newname)))
      (setq count (1- count)))
    newname))

(defun mew-summary-folder-dir-newp ()
  ;; buffer switched
  (let* ((dir (mew-file-chase-links (mew-expand-file-name (buffer-name))))
	 (tdir (nth 5 (file-attributes dir)))
	 (da (car tdir))
	 (db (car (cdr tdir)))
	 (cache (expand-file-name mew-summary-cache-file dir))
	 (tcache (nth 5 (file-attributes cache)))
	 (fa (car tcache))
	 (fb (car (cdr tcache))))
    (cond
     ((null tdir) nil)
     ((null tcache) t) ;; no cache, do update!
     ((> da fa) t)
     ((= da fa) (if (> db fb) t nil)) ;; nil if same 
     (t nil)
     )
    ))

(defun mew-summary-folder-cache-oldp ()
  ;; buffer switched
  (let* ((tfile (nth 5 (file-attributes 
			(mew-expand-file-name mew-summary-cache-file 
					      (buffer-name)))))
	 (fa (car tfile))
	 (fb (car (cdr tfile)))
	 (tbuf mew-summary-buffer-folder-cache-time)
	 (ba (car tbuf))
	 (bb (car (cdr tbuf))))
    (cond
     ((or (null tfile) (null tbuf)) nil) ;; xxx
     ((> fa ba) t)
     ((= fa ba) (if (> fb bb) t nil)) ;; nil if same 
     (t nil)
     )
    ))

(defun mew-summary-folder-cache-manage (folder)
  (switch-to-buffer folder)
  (let ((cache (mew-expand-file-name mew-summary-cache-file folder)))
    (if mew-summary-cache-use
        (if (and (file-exists-p cache) (mew-summary-folder-cache-oldp))
            (let ((buffer-read-only nil))
              (erase-buffer)
              (insert-file-contents cache)
              (setq mew-summary-buffer-folder-cache-time 
                    (nth 5 (file-attributes cache)))
              ))))
  (if (not (equal major-mode 'mew-summary-mode)) (mew-summary-mode))
  )

;;; xxx
;(defun mew-summary-folder-cache-manage (folder)
;  (switch-to-buffer folder)
;  (let ((cache (mew-expand-file-name mew-summary-cache-file folder)))
;    (if mew-summary-cache-use
;	(if (file-exists-p cache)
;	    (if (mew-summary-folder-cache-oldp)
;		(let ((buffer-read-only nil))
;		  (erase-buffer)
;		  (insert-file-contents cache)
;		  (setq mew-summary-buffer-folder-cache-time 
;			(nth 5 (file-attributes cache)))
;		  ))
;	  ;; prevent that Mew asks "update?" every inc when 
;	  ;; .mew-cache doesn't exist in +inbox.
;	  (mew-summary-folder-cache-save))))
;  (if (not (equal major-mode 'mew-summary-mode)) (mew-summary-mode))
;  )

(defun mew-summary-folder-cache-save ()
  (let ((cache (mew-expand-file-name mew-summary-cache-file (buffer-name))))
    (if (file-writable-p cache)
        (progn
          (write-region (point-min) (point-max) cache)
          (setq mew-summary-buffer-folder-cache-time
                (nth 5 (file-attributes cache)))
          ))
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; show
;;;

(defun mew-summary-highlight-lines-region (beg end &optional move)
  (if (and mew-highlight-lines-use mew-emacs-p)
      (let ((p (point))
	    (buffer-read-only nil))
	(goto-char beg)
	(if move
	    (let ((ovls (overlays-at (point))))
	      (mapcar 
	       (function 
		(lambda (ovl) (move-overlay ovl end
					    (overlay-end ovl))))
	       ovls)))
	(while (> end (point))
	  (overlay-put
	   (make-overlay (point) (progn (end-of-line) (point)))
	   'mouse-face 'highlight)
	  (forward-line))
	(goto-char p))
    )
  )

(defun mew-summary-underline-lines ()
  (if mew-underline-lines-use
      (progn
	(if (mew-overlayp mew-underline-overlay)
	    (mew-move-overlay mew-underline-overlay
			      (save-excursion (beginning-of-line) (point))
			      (save-excursion (end-of-line) (point)))
	  (setq mew-underline-overlay
		(mew-make-overlay 
		 (save-excursion (beginning-of-line) (point))
		 (save-excursion (end-of-line) (point)))))
	(mew-overlay-put mew-underline-overlay 'face 'underline))))

(defun mew-message-highlight-url ()
  (let ((url-regex "\\b\\(s?https?\\|ftp\\|file\\|gopher\\|news\\|telnet\\|wais\\|mailto\\):\\(//[-a-zA-Z0-9_.]+:[0-9]*\\)?[-a-zA-Z0-9_=?#$@~`%&*+|\\/.,]*[-a-zA-Z0-9_=#$@~`%&*+|\\/]")
	overlay)
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward url-regex mew-bold-url-size t)
	(setq overlay (mew-make-overlay (match-beginning 0) (match-end 0)))
	(mew-overlay-put overlay 'face 'bold)
	(mew-overlay-put overlay 'mouse-face 'highlight)
	))))

(defun mew-summary-toggle-analysis ()
  "Toggle with/out MIME analysis."
  (interactive)
  (cond 
   (mew-analysis
    (message "Skip MIME analysis")
    (setq mew-analysis nil))
   (t
    (message "Display message with MIME analysis")
    (setq mew-analysis t))
   )
  (mew-cache-flush)
  )

(defun mew-summary-show ()
  (interactive)
  (let* ((fld (mew-summary-folder-name))
	 (msg (mew-summary-message-number))
	 (ofld-msg (mew-current-get 'message))
	 (part (mew-summary-part-number))
	 (opart (mew-current-get 'part))
	 (buf (buffer-name))
	 (next nil))
    (if (not (file-exists-p (mew-expand-file-name (concat fld "/" msg))))
	(message "File does not exist.")
      (mew-summary-toggle-disp-msg 'on)
      (unwind-protect
	  (progn
	    ;; message buffer
	    (cond 
	     (msg 
	      (mew-window-configure buf 'message)
	      (if (or (null ofld-msg)
		      (null (equal (cons fld msg) ofld-msg))
		      (and (equal (cons fld msg) ofld-msg) 
			   (null (equal opart part))))
		  (mew-summary-display-message fld msg buf)
		(if (mew-message-next-page)
		    (setq next t)))
	      )
	     (part
	      (mew-window-configure buf 'message)
	      (if (or (null opart) (null (equal opart part)))
		  (mew-summary-display-part 
		   (mew-cache-decode-syntax (mew-cache-hit ofld-msg)) part)
		(if (mew-message-next-page)
		    (setq next t)))
	      )
	     (t
	      (message "No message")
	      nil
	      )
	     ))
	(if (or msg part)
	    (progn
	      (pop-to-buffer buf)
	      (mew-summary-recenter)
	      (mew-summary-underline-lines)
	      (if next
		  (cond 
		   ((equal mew-summary-show-direction 'down)
		    (mew-summary-display-down))
		   ((equal mew-summary-show-direction 'up)
		    (mew-summary-display-up))
		   ((equal mew-summary-show-direction 'next)
		    (mew-summary-display-next))
		   (t ())
		   )
		)
	      )
	  )
	))
    )
  (set-buffer-modified-p nil)
  )

(defun mew-summary-toggle-disp-msg (&optional arg)
  (interactive)
  (cond 
   ((equal arg 'on)
    (setq mew-summary-buffer-disp-msg t))
   ((equal arg 'off)
    (setq mew-summary-buffer-disp-msg nil))
   (t
    (setq mew-summary-buffer-disp-msg (not mew-summary-buffer-disp-msg))
    (if mew-summary-buffer-disp-msg
	(mew-summary-display)
      (mew-decode-syntax-delete) ;; xxx 
      (mew-window-configure (current-buffer) 'summary)))))

(defun mew-summary-display (&optional notforce)
  (interactive)
  (let* ((fld (mew-summary-folder-name))
	 (msg (mew-summary-message-number))
	 (ofld-msg (mew-current-get 'message))
	 (part (mew-summary-part-number))
	 (opart (mew-current-get 'part))
	 (buf (buffer-name))
	 (cache nil))
    (cond 
     ((and (null msg) (null part))
      (message "No message"))
     ((not (file-exists-p (mew-expand-file-name (concat fld "/" msg))))
      (message "File does not exist"))
     (t
      (unwind-protect
	  (progn
	    (if (or (interactive-p) mew-summary-buffer-disp-msg)
		(progn 
		  (mew-summary-toggle-disp-msg 'on)
		  (mew-window-configure buf 'message))
	      (mew-window-configure (current-buffer) 'summary)
	      (set-buffer (mew-buffer-message)))
	    (cond
	     (msg
	      (if (or (null notforce)
		      (null ofld-msg)
		      (null (equal (cons fld msg) ofld-msg)))
		  (if (or (interactive-p) mew-analysis)
		      (setq cache
			    (mew-summary-display-message
			     fld msg buf 'analysis))
		    (setq cache
			  (mew-summary-display-message fld msg buf))))
	      )
	     (part
	      (if (or (null opart) (null (equal opart part)))
		  (mew-summary-display-part 
		   (mew-cache-decode-syntax (mew-cache-hit ofld-msg)) part)))
	     (t (message "No message"))
	     ))
	(progn
	  (pop-to-buffer buf)
	  (if (null notforce) (mew-summary-recenter))
	  (mew-summary-underline-lines)
	  )
	)
      (set-buffer-modified-p nil)
      (or cache (mew-cache-hit ofld-msg)) ;; return value
      )
     )
    ))

(defun mew-summary-mode-line (buf)
  (save-excursion
    (set-buffer buf)
    ;; if not running process in this buffer
    ;; display how many messages are unread
    (if (null mew-summary-buffer-process)
	(setq mew-summary-buffer-left-msgs ;; local variable
	      (int-to-string (1- (count-lines (point) (point-max)))))
      )
    ))

(defun mew-summary-display-message (fld msg buf &optional analysis)
  ;; message buffer
  (let ((hit nil) (zmacs-regions nil) (buffer-read-only nil))
    (erase-buffer) ;; for PGP pass phrase input
    (mew-decode-syntax-delete)
    (mew-summary-mode-line buf)
    (set-buffer-modified-p nil)
    (set-marker (mark-marker) nil) ;; kill mark for cite
    (mew-current-set 'message (cons fld msg))
    (mew-current-set 'part nil)
    (mew-current-set 'cache nil)
    (setq mew-decode-result nil)
    (setq mew-decode-syntax nil)
    (cond
     ((equal fld mew-draft-folder)
      (insert-file-contents
       (mew-expand-file-name (concat fld "/" msg)))
      )
     ((not (or mew-analysis analysis))
      (insert-file-contents
       (mew-expand-file-name (concat fld "/" msg)))
      (mew-header-arrange)
      )
     (t
      (setq hit (mew-cache-message (cons fld msg)))
      (mew-current-set 'cache hit)
      (setq mew-decode-syntax (mew-cache-decode-syntax hit))
      (setq mew-decode-error  (mew-cache-decode-error hit))
      (if (mew-syntax-multipart-p (mew-syntax-get-part mew-decode-syntax))
	  (mew-decode-syntax-print msg mew-decode-syntax buf)
	(mew-decode-syntax-set-privacy mew-decode-syntax))
      (mew-mime-message/rfc822 mew-decode-syntax)
      (if mew-decode-error (message "MIME decoding error: %s"mew-decode-error))
      )
     )
    (setq mew-decode-result nil)
    (if mew-emacs-p (mew-message-highlight-url))
    (run-hooks 'mew-message-hook)
    (set-buffer-modified-p nil)
    hit ;; return value
    ))

(defun mew-summary-display-part (fullpart num &optional non-erase execute)
;; called in message buffer
;; return t to next part ;; xxx ???
  (let* ((syntax  (mew-syntax-get-entry-strnum fullpart num))
	 (begin   (mew-syntax-get-begin  syntax))
	 (end     (mew-syntax-get-end    syntax))
	 (ctl     (mew-syntax-get-ct syntax))
	 (ct      (car ctl))
	 (params  (cdr ctl))
	 (attr    (mew-content-attr ct mew-mime-content-type))
	 (program (mew-content-program attr))
	 (options (mew-content-options attr))
	 (async   (mew-content-async   attr))
	 (zmacs-regions nil)  ;; for XEmacs
	 file)
    (if (null non-erase)
	(let ((buffer-read-only nil))
	  (erase-buffer)
	  (set-buffer-modified-p nil))) ;; xxx
    (set-marker (mark-marker) nil) ;; kill mark for cite
    (setq mew-message-citation 'noheader)
    ;;
    (if (symbolp program)
	(if (fboundp program)
	    (cond ((eq program 'mew-mime-message/rfc822)
		   (funcall program syntax)) ;; for recursive MIME
		  ((eq program 'mew-mime-external-body)
		   (funcall program begin end params execute))
		  (t
		   (funcall program begin end params))))
      (save-excursion
	(set-buffer (mew-current-get 'cache))
	(if (not (mew-which program exec-path))
	    (message "Program %s is not found" program)
	  (if (not (or mew-icon-p
		       ;; gee, the following has a bug that multipart
		       ;; icons disappear from toolbar...
		       ;;(and mew-icon-p (button-event-p last-command-event))
		       (mew-y-or-n-p (format "Start %s? " program))))
	      ()
	    (if (not (file-exists-p mew-temp-dir))
		(mew-make-directory mew-temp-dir)) ;; just in case
	    (setq file (make-temp-name mew-temp-file))
	    ;; NEVER use call-process-region for privary reasons
	    (mew-flet
	     (write-region begin end file))
	    (if async
		(mew-mime-start-process program options file)
	      (mew-mime-call-process program options file))
	    ))))
    (mew-current-set 'part num) ;; should be after funcall
    (if non-erase
	()
      (if mew-emacs-p (mew-message-highlight-url))
      (run-hooks 'mew-message-hook)
      (set-buffer-modified-p nil) ;; xxx message buffer
      )
    ))

(defun mew-summary-execute-external ()
  (interactive)
  (let* ((ofld-msg (mew-current-get 'message))
	 (msg (mew-summary-message-number))
	 (part (mew-summary-part-number))
	 (buf (buffer-name)))
    (if (or msg (not part))
	(message "Not a part")
      (unwind-protect
	  (progn
	    (mew-summary-toggle-disp-msg 'on)
	    (mew-window-configure buf 'message)
	    (set-buffer (mew-buffer-message))
	    (mew-summary-display-part
	     (mew-cache-decode-syntax (mew-cache-hit ofld-msg)) part nil t))
	(pop-to-buffer buf)
	(mew-summary-underline-lines)))))

(defun mew-summary-recenter ()
  (interactive)
  (if (or mew-summary-recenter-p
	  (interactive-p))
      (recenter (/ (- (window-height) 2) 2))
    ))

(defun mew-summary-display-top ()
  (interactive)
  (goto-char (point-min))
  (mew-summary-display))

(defun mew-summary-display-bottom ()
  (interactive)
  (goto-char (point-max))
  (if (not (bobp)) (forward-line -1))
  (mew-summary-display))


(defun mew-summary-mouse-show (e)
  (interactive "e")
  (mouse-set-point e)
  (beginning-of-line)
  (mew-summary-show))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; direction
;;;

(defun mew-summary-next ()
  (if (equal mew-summary-buffer-direction 'up)
      (mew-summary-up)
    (mew-summary-down))
  )

(defun mew-summary-message-next ()
  (if (equal mew-summary-buffer-direction 'up)
      (mew-summary-message-up)
    (mew-summary-message-down))
  )

(defun mew-summary-down ()
  (forward-line)
  (cond 
   ((re-search-forward 
     (concat mew-summary-message-regex 
	     "[ " (char-to-string mew-mark-hop) "]\\|"
	     mew-summary-part-regex)
     nil t nil)
    (beginning-of-line)
    (setq mew-summary-buffer-direction 'down)
    t)
   (t 
    (mew-decode-syntax-delete)
    (mew-current-set 'message nil)
    (forward-line -1)
    (mew-window-configure (current-buffer) 'summary)
    (message "No more message")
    nil)
   )
  )

(defun mew-summary-message-down ()
  (forward-line)
  (cond 
   ((re-search-forward
     (concat mew-summary-message-regex "[ " (char-to-string mew-mark-hop) "]")
     nil t nil)
    (beginning-of-line)
    (setq mew-summary-buffer-direction 'down)
    t)
   (t 
    (message "No more message")
    nil)
   )
  )

(defun mew-summary-up ()
  (interactive)
  (cond 
   ((re-search-backward
     (concat mew-summary-message-regex 
	     "[ " (char-to-string mew-mark-hop) "]\\|"
	     mew-summary-part-regex)
     nil t nil)
    (setq mew-summary-buffer-direction 'up)
    t)
   (t 
    (mew-decode-syntax-delete)
    (mew-window-configure (current-buffer) 'summary)
    (message "No more message")
    nil)
   ))

(defun mew-summary-message-up (&optional nochange mark)
  (interactive)
  (let ((regex (concat " "
		       (char-to-string mew-mark-hop)
		       (if mark (char-to-string mark)))))
    (setq regex (concat mew-summary-message-regex "[" regex "]"))
    (cond 
     ((re-search-backward regex nil t nil)
      (if (null nochange)
	  (setq mew-summary-buffer-direction 'up))
      t)
     (t 
      (message "No more message")
      nil)
     ))
  )

(defun mew-summary-display-next ()
  (interactive)
  (cond
   ((and (mew-summary-next) mew-summary-buffer-disp-msg)
    (mew-summary-display)
    ))
  )

(defun mew-summary-display-down ()
  (interactive)
  (if (and (mew-summary-down) mew-summary-buffer-disp-msg)
      (mew-summary-display))
  )

(defun mew-summary-down-mark (mark)
  (interactive)
  (forward-line)
  (cond 
   ((re-search-forward (concat mew-summary-message-regex
			       (regexp-quote (char-to-string mark)))
		       nil t nil)
    (beginning-of-line)
    t)
   (t 
    (forward-line -1)
    (message "No more marked message")
    nil)))

(defun mew-summary-display-hop-down ()
  (interactive)
  (let ((part nil))
    (if (looking-at mew-summary-part-regex)
	(setq part t)
      (save-excursion
	(forward-line)
	(if (looking-at mew-summary-part-regex)
	    (setq part t))))
    (if (null part)
	(if (mew-summary-down-mark mew-mark-hop)
	    (if mew-summary-buffer-disp-msg ;; just seed up!
		(mew-summary-display)))
      (mew-summary-message-down)
      (if mew-summary-buffer-disp-msg ;; just seed up!
	  (mew-summary-display)))
    ))

(defun mew-summary-display-up ()
  (interactive)
  (if (and (mew-summary-up) mew-summary-buffer-disp-msg)
      (mew-summary-display))
  )

(defun mew-summary-up-mark (mark)
  (interactive)
  (cond 
   ((re-search-backward (concat mew-summary-message-regex
			       (regexp-quote (char-to-string mark)))
			nil t nil)
    t)
   (t 
    (message "No more marked message")
    nil)
   ))

(defun mew-summary-display-hop-up ()
  (interactive)
  (let ((part nil))
    (if (looking-at mew-summary-part-regex)
	(setq part t))
    (if (null part)
	(if (mew-summary-up-mark mew-mark-hop)
	    (if mew-summary-buffer-disp-msg ;; just seed up!
		(mew-summary-display)))
      (mew-summary-message-up)
      (if mew-summary-buffer-disp-msg ;; just seed up!
	  (mew-summary-display)))
    ))

(defun mew-summary-prev-page ()
  (interactive)
  (let ((fld (mew-summary-folder-name))
	(msg (mew-summary-message-number))
	(part (mew-summary-part-number)))
    (cond
     ((not (or msg part))
      (message "No message or part here"))
     ((and msg (not (equal (cons fld msg) (mew-current-get 'message))))
      (message "Please show the current message first"))
     ;; part exist only when msg = current msg
     ((and part (not (equal part (mew-current-get 'part)))) 
      (message "Please show the current part first"))
     (t 
      (let ((buf (current-buffer)))
	(unwind-protect
	    (progn
	      (pop-to-buffer (mew-buffer-message))
	      (mew-message-prev-page))
	  (pop-to-buffer buf))))
     )))

(defun mew-summary-scroll-up ()
  (interactive)
  (let ((buf (current-buffer))
	(msg (mew-summary-message-number))
	(ofld-msg (mew-current-get 'message))
	(part (mew-summary-part-number))
	(opart (mew-current-get 'part)))
    (cond ((or (and msg (null part) (string= msg (cdr ofld-msg)))
	       (and part (string= part opart)))
	   (unwind-protect
	       (progn
		 (mew-window-configure buf 'message)
		 (mew-message-next-page 1))
	     (pop-to-buffer buf)))
	  ((or msg part)
	   (mew-summary-show))
	  (t
	   (message "No message or part here")))))

(defun mew-summary-scroll-down ()
  (interactive)
  (if (or (mew-summary-message-number) (mew-summary-part-number))
      (let ((buf (current-buffer)))
	(unwind-protect
	    (progn
	      (mew-window-configure buf 'message)
	      (mew-message-prev-page 1)
	      )
	  (pop-to-buffer buf))
	)
    (message "No message or part here")
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; to Draft mode
;;;

(defun mew-summary-send (&optional unconfig to cc subject)
  (interactive)
  (let ((file (mew-draft-get-new)))
    (if (null unconfig)
	(mew-current-set 'window (current-window-configuration)))
    (mew-window-configure (current-buffer) 'draft)
    (unwind-protect
	(progn
	  (switch-to-buffer (find-file-noselect file))
	  (mew-draft-rename file)
	  (mew-draft-header subject nil to nil)
	  (mew-draft-mode) ;; hack hack mew-draft-buffer-header
	  )
      (sit-for 0)  ;; force to redisplay before save-buffer
      (save-buffer))) ;; to make sure not to use this draft again
  (message "Draft is prepared")
  )

(defun mew-summary-reply ()
  (interactive)
  (cond 
   ((or (mew-summary-message-number)
	(mew-summary-part-number))
    (mew-current-set 'window (current-window-configuration))
    (let ((buf (buffer-name))
	  (file (mew-draft-get-new))
	  from to cc newsgroups subject in-reply-to references
	  cbuf cache)
      (unwind-protect
	  (progn
	    (if (get-buffer (mew-buffer-message))
		(delete-windows-on (mew-buffer-message)))
	    (if (< (window-height) 25) (delete-other-windows))
	    (let ((split-window-keep-point t))
	      (split-window-vertically))
	    (switch-to-buffer-other-window (find-file-noselect file))
	    (mew-draft-rename file)
	    (setq cbuf (current-buffer)) ;; draft
	    (pop-to-buffer buf) 
	    ;; summary or virtual
	    ;;
;;	    (setq cache (mew-summary-display)) ;; force to display
;; to reply scrambled mail. Is this okay?
	    (setq cache (mew-summary-display 'notforce))
	    (if cache
		(set-buffer cache)
	      (set-buffer (mew-buffer-message)))
	    ;; now cache buffer
	    (setq from (mew-header-address-collect '("From:")))
	    (cond 
	     ;; This message was sent by me.
	     ;; So, maintain To: and Cc:.
	     ((and from (string-match (concat "^" mew-mail-address "$")
				      (car from)))
	      (setq to (mew-header-address-collect '("To:" "Apparently-To:")))
 	      (if (and (null (cdr to))
		       (car to) ;; not null
		       (string-match ":;" (car to)))
		  (setq to (or (mew-header-address-collect '("Reply-To:"))
			       from)))
	      (setq cc (mew-header-address-collect '("Cc:"))))
	     ;;
	     (t
	      (cond 
	       ((mew-header-get-value "Reply-To:")
		(setq to (mew-header-address-collect mew-replyto-to-list))
		(setq cc (mew-header-address-collect mew-replyto-cc-list)))
	       (t
		(setq to (mew-header-address-collect mew-noreplyto-to-list))
		(setq cc (mew-header-address-collect mew-noreplyto-cc-list)))
	       )
	      )
	     )
	    (setq newsgroups (or (mew-header-get-value "Followup-To:")
				 (mew-header-get-value "Newsgroups:")))
	    (setq subject (mew-header-get-value "Subject:"))
	    (if (and subject (not (string-match "^Re:" subject)))
		(setq subject (concat "Re: " subject)))
	    (setq in-reply-to (mew-header-get-value "Date:"))
	    (setq references (mew-header-get-value "Message-ID:"))
	    ;;
	    (pop-to-buffer cbuf) ;; draft
	    (mew-draft-header subject nil to cc newsgroups in-reply-to references)
	    (mew-draft-mode) ;; for hilight
	    (sit-for 0)
	    )
	(save-buffer))) ;; to make sure not to use this draft again
    (message "Draft is prepared")
    t) ;; preparation is succeeded.
   (t 
    (message "No message")
    nil) ;; preparation is failed.
   )
  )

(defun mew-summary-reply-with-citation ()
  "Reply to current message and insert it in draft-buffer."
  (interactive)
  (if (mew-summary-reply)
      (mew-draft-cite)
    ))

(defun mew-summary-forward ()
  (interactive)
  (if (not (or (mew-summary-message-number) (mew-summary-part-number)))
      (message "No message")
    (mew-current-set 'window (current-window-configuration))
    (if (mew-summary-part-number)
	(progn
	  (mew-summary-message-up t) ;; no direction change
	  (mew-summary-display) ; xxxxx
	  )
      )
    (let* ((buf (buffer-name))
	   (fld (mew-summary-folder-name))
	   (msg (mew-summary-message-number))
	   (cbuf nil)
	   (subject "")
	   (file (mew-draft-get-new))
	   (dirname (file-name-nondirectory file)))
      (unwind-protect
	  (progn
	    (delete-other-windows)
	    (let ((split-window-keep-point t))
	      (split-window-vertically))
	    (switch-to-buffer-other-window (find-file-noselect file))
	    (mew-draft-rename file)
	    (setq cbuf (current-buffer)) ;; draft
	    (pop-to-buffer buf)
	    ;;
	    (let ((mew-analysis t))
	      (set-buffer (mew-summary-display))) ;; force to display
	    ;; now cache buffer
	    (setq subject (concat "Forward: "
				  (mew-header-get-value "Subject:")))
	    (pop-to-buffer cbuf) ;;; draft
	    ;;
	    (mew-draft-header subject 'nl)
	    (mew-draft-mode)
	    (mew-draft-multi-copy file (list (cons fld msg)))
	    (setq mew-encode-syntax
		  (mew-encode-syntax-initial-multi dirname 1))
	    (save-excursion
	      (mew-draft-prepare-attachments)
	      )
	    ;; XEmacs doesn't draw attachments unless sit for 0...
	    (sit-for 0) 
	    ;; XEmacs doesn't draw toolbar, so...
	    (if mew-icon-p
		(progn
		  (set-specifier default-toolbar-visible-p nil)
		  (set-specifier default-toolbar-visible-p t)
		  ))
	    )
	(save-buffer))) ;; to make sure not to use this draft again
    (message "Draft is prepared"))
  )

(defun mew-summary-multi-forward ()
  (interactive)
  (let ((fld-msg (mew-summary-mark-collect2 mew-mark-mark)))
    (cond
     (fld-msg
      (mew-current-set 'window (current-window-configuration))
      (let* ((file (mew-draft-get-new))
	     (dirname (file-name-nondirectory file)))
	(unwind-protect
	    (progn
	      (delete-other-windows)
	      (let ((split-window-keep-point t))
		(split-window-vertically))
	      (switch-to-buffer-other-window (find-file-noselect file))
	      (mew-draft-rename file)
	      (mew-draft-header nil 'nl)
	      (mew-draft-mode)
	      (mew-draft-multi-copy file fld-msg)
	      (setq mew-encode-syntax
		    (mew-encode-syntax-initial-multi dirname (length fld-msg)))
	      (save-excursion
		(mew-draft-prepare-attachments)
		)
	      ;; XEmacs doesn't draw attachments unless sit for 0...
	      (sit-for 0) 
	      ;; XEmacs doesn't draw toolbar, so...
	      (if mew-icon-p
		  (progn
		    (set-specifier default-toolbar-visible-p nil)
		    (set-specifier default-toolbar-visible-p t)
		    ))
	      )
	  (save-buffer)) ;; to make sure not to use this draft again
	(message "Draft is prepared"))
      )
     (t (message "No marks"))
     )
    ))

(defun mew-draft-multi-copy (draft fld-msg-list)
  (let* ((mimefolder (mew-draft-to-mime draft))
	 (mimedir (mew-expand-file-name mimefolder)))
    (if (null (file-directory-p mimedir))
	(mew-make-directory mimedir)
      (if (null (mew-directory-empty-p mimedir))
	  (if (mew-y-or-n-p (format "Mime folder %s is not empty. Delete it? "
				mimefolder))
	      (progn
		(mew-delete-directory-recursively mimedir)
		(mew-make-directory mimedir))
	    )))
    (while fld-msg-list
      (make-symbolic-link (mew-expand-file-name
			   (cdr (car fld-msg-list)) ;; message
			   (car (car fld-msg-list))) ;; folder
			  (mew-draft-get-new mimefolder))
      (setq fld-msg-list (cdr fld-msg-list)))
    ))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Edit again
;;;

(defun mew-summary-reedit ()
  "Reedit a message in a folder or a message encapsulated in MIME."
  (interactive)
  (let* ((msg (mew-summary-message-number)) ;; must get msg here
	(part (mew-summary-part-number))
	(fld (mew-summary-folder-name))
	(rename nil)
	(beg)
	(cache (if part (mew-current-get 'cache)))
	(syntax (if part (mew-syntax-get-entry-strnum
			  (mew-cache-decode-syntax cache) part)))
	(ct (if syntax (capitalize (car (mew-syntax-get-ct syntax))))))
    (cond
     ((and (null msg) (null part))
      (message "No message"))
     ((and part (not (equal (capitalize mew-ct-msg) ct)))
      (message "Can't reedit %s" ct))
     (t
      (mew-current-set 'window (current-window-configuration))
      (mew-window-configure (current-buffer) 'summary)
      (unwind-protect
	  (progn
	    (cond
	     ((or part (not (equal fld mew-draft-folder)))
	      (setq rename (mew-draft-get-new)))
	     (t 
	      (setq rename msg))
	     )
	    ;; prepare draft file
	    (switch-to-buffer 
	     (find-file-noselect 
	      (mew-expand-file-name rename mew-draft-folder)))
	    (cond
	     (part
	      (insert-buffer-substring
	       cache
	       (mew-syntax-get-begin syntax)
	       (mew-syntax-get-end (mew-syntax-get-part syntax)))) ;;; xxxx
	     ((not (equal fld mew-draft-folder))
	      (insert-file-contents (mew-expand-file-name msg fld))
	      )
	     ;; if fld equal mew-draft-folder, message already exists.
	     )
	    (mew-draft-rename rename)
	    (mew-header-delete-lines mew-field-delete)
	    ;; delimiter setup
	    ;; Both "^$" and "----" are OK.
	    (goto-char (point-min))
	    (re-search-forward mew-eoh2 nil t)
	    (beginning-of-line)
	    (setq beg (point))
	    (forward-line)
	    (delete-region beg (point))
	    ;; Dcc or Fcc:
	    (if (and mew-fcc (not (mew-header-get-value "Fcc:")))
		(mew-header-insert-here "Fcc:" mew-fcc))
	    (if (and mew-dcc (not (mew-header-get-value "Dcc:"))) 
		(mew-header-insert-here "Dcc:" mew-dcc))
 	    (if (and mew-from (not (mew-header-get-value "From:")))
 		(mew-header-insert-here "From:" mew-from))
 	    (if (and mew-reply-to (not (mew-header-get-value "Reply-To:")))
 		(mew-header-insert-here "Reply-To:" mew-reply-to))
	    (if (and mew-x-mailer (not (mew-header-get-value "X-Mailer:")))
		(mew-header-insert-here "X-Mailer:" mew-x-mailer))
	    (setq mew-draft-buffer-header (point-marker))
	    (insert mew-header-separator "\n")
	    (mew-draft-mode)
	    )
	(save-buffer))
      (message "Draft is prepared"))
     )
    ))

(defun mew-summary-edit-again ()
  (interactive)
  (let ((msg (mew-summary-message-number)) ;; must get msg here
	(fld (mew-summary-folder-name))
	(rename nil)
	beg)
    (cond 
     ((eobp) (message "No message"))
     ((null msg)
      (message "Please use this command on a message"))
     (t
      (mew-current-set 'window (current-window-configuration))
      (mew-window-configure (current-buffer) 'summary)
      (unwind-protect
	  (progn
	    (cond
	     ((not (equal fld mew-draft-folder))
	      (setq rename (mew-draft-get-new)))
	     (t 
	      (setq rename msg))
	     )
	    ;; prepare draft file
	    (switch-to-buffer 
	     (find-file-noselect 
	      (mew-expand-file-name rename mew-draft-folder)))
	    (cond
	     ((not (equal fld mew-draft-folder))
	      (insert-file-contents (mew-expand-file-name msg fld))
	      )
	     ;; if fld equal mew-draft-folder, message already exists.
	     )
	    (mew-draft-rename rename)
	    (goto-char (point-min))
	    (if (re-search-forward mew-summary-edit-again-regex nil t)
		(progn
		  (forward-line)
		  ;; skip blank lines
		  (while (looking-at "^$") (forward-line))
		  (delete-region (point-min) (point))
		  ))
	    (mew-header-delete-lines mew-field-delete)
	    ;; delimiter setup
	    ;; Both "^$" and "----" are OK.
	    (goto-char (point-min))
	    (re-search-forward mew-eoh2 nil t)
	    (beginning-of-line)
	    (setq beg (point))
	    (forward-line)
	    (delete-region beg (point))
	    ;; Dcc or Fcc:
	    (if (and mew-fcc (not (mew-header-get-value "Fcc:")))
		(mew-header-insert-here "Fcc:" mew-fcc))
	    (if (and mew-dcc (not (mew-header-get-value "Dcc:"))) 
		(mew-header-insert-here "Dcc:" mew-dcc))
 	    (if (and mew-from (not (mew-header-get-value "From:")))
 		(mew-header-insert-here "From:" mew-from))
 	    (if (and mew-reply-to (not (mew-header-get-value "Reply-To:")))
 		(mew-header-insert-here "Reply-To:" mew-reply-to))
	    (if (and mew-x-mailer (not (mew-header-get-value "X-Mailer:")))
		(mew-header-insert-here "X-Mailer:" mew-x-mailer))
	    (setq mew-draft-buffer-header (point-marker))
	    (insert mew-header-separator "\n")
	    (mew-draft-mode)
	    )
	(save-buffer))
      (message "Draft is prepared"))
     )
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Sorting and Packing
;;;

(defun mew-summary-sort ()    
  (interactive)
  (let ((folder (buffer-name)))
    (if (null (mew-summary-exclusive-p))
	()
      (if (mew-summary-mark-exist-p (list mew-mark-rmm mew-mark-refile))
	  (if (mew-y-or-n-p
	       "Marked messages exist. Process mark before scan? ")
	      (mew-summary-exec)))
      (if (null (mew-y-or-n-p (format "Sort %s ? " folder)))
	  ()
	(setq mew-summary-buffer-process t)
	(message "Sorting %s ... " folder)
	(call-process mew-prog-sortm nil nil nil folder)
	(message "Sorting %s ... done" folder)
	(setq mew-summary-buffer-process nil)
	(let ((buffer-read-only nil)) (erase-buffer))  ;; for update
	(mew-summary-scan-body folder (mew-input-range folder))
	)
      )
    ))

(defun mew-summary-pack ()    
  (interactive)
  (let ((folder (buffer-name)))
    (if (null (mew-summary-exclusive-p))
	()
      (if (mew-summary-mark-exist-p (list mew-mark-rmm mew-mark-refile))
	  (if (mew-y-or-n-p
	       "Marked messages exist. Process mark before scan? ")
	      (mew-summary-exec)))
      (if (null (mew-y-or-n-p (format "Pack %s? " folder)))
	  ()
	(setq mew-summary-buffer-process t)
	(message "Packing %s ... " folder)
	(call-process mew-prog-pack nil nil nil folder "-pack")
	(message "Packing %s ... done" folder)
	(setq mew-summary-buffer-process nil)
	(let ((buffer-read-only nil)) (erase-buffer)) ;; for update
	(mew-summary-scan-body folder (mew-input-range folder))
	)
      )
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Good old days...
;;;

(defun mew-summary-unshar ()
  (interactive)
  (let ((fld-msg (nreverse (mew-summary-mark-collect2 mew-mark-mark)))
	(files nil)
	(dir nil))
    (if (null fld-msg)
	(message "No marked messages")
      (if (null (mew-y-or-n-p (format "Execute unshar for these messages? "
				      mew-prog-unshar)))
	  ()
	(setq dir (mew-input-directory-name))
	(while fld-msg
	  (setq files (cons (mew-expand-file-name
			     (cdr (car fld-msg)) ;; message
			     (car (car fld-msg))) ;; folder
			    files))
	  (setq fld-msg (cdr fld-msg))
	  )
	(message "Executing %s ... " mew-prog-unshar)
	(apply (function call-process) 
	       mew-prog-unshar nil nil nil "-d" dir files)
	(message "Executing %s ... done" mew-prog-unshar)
	))
    ))

(defun mew-summary-uudecode ()
  (interactive)
  (let ((fld-msg (nreverse (mew-summary-mark-collect2 mew-mark-mark)))
	(files nil)
	(dir nil)
	(tarfile nil)
	(case-fold-search nil))
    (if (null fld-msg)
	(message "No marked messages")
      (if (null (mew-y-or-n-p (format "Execute %s for these messages? "
				   mew-prog-uumerge)))
	  ()
	(setq dir (mew-input-directory-name))
	(while fld-msg
	  (setq files (cons (mew-expand-file-name
			     (cdr (car fld-msg)) ;; message
			     (car (car fld-msg))) ;; folder
			    files))
	  (setq fld-msg (cdr fld-msg))
	  )
	(save-excursion
	  (set-buffer mew-buffer-tmp)
	  (let ((buffer-read-only nil)) (erase-buffer))
	  (message "Executing %s ..." mew-prog-uumerge)
	  (apply (function call-process) 
		 mew-prog-uumerge nil t nil "-d" dir files)
	  (message "Executing %s ... done" mew-prog-uumerge)
	  (goto-char (point-min))
	  (if (looking-at "^uumerge:")
	      (message "Failed to executing %s" mew-prog-uumerge)
	    (forward-line)
	    (setq tarfile (mew-buffer-substring (point-min) (1- (point))))
	    (setq tarfile (mew-summary-prog-exec
			   mew-prog-compress "-df" "Z" dir tarfile))
	    (setq tarfile (mew-summary-prog-exec
			   mew-prog-gzip "-df" "gz" dir tarfile))
	    (if (string-match "^\\(.*\\)\\.tar$" tarfile)
		(if (null (mew-y-or-n-p (format "Execute %s for %s? " 
					    mew-prog-tar tarfile)))
		    ()
		  (message (format "Executing %s for %s ... "
				    mew-prog-tar tarfile))
		  (call-process mew-prog-shell nil nil nil
				mew-prog-shell-arg
				(concat "cd " dir " ; cat " 
					tarfile " | tar -xf -"))
		  (message (format "Executing %s for %s ... done"
				   mew-prog-tar tarfile))
		  ))
	    ))
	))
    ))

(defun mew-summary-prog-exec (prog opts suf dir tarfile)
  (if (string-match (format "^\\(.*\\)\\.%s$" suf) tarfile)
      (if (null (mew-y-or-n-p (format "Execute %s for %s? " prog tarfile)))
	  tarfile
	(message (format "Executing %s for %s ... " prog tarfile))
	(call-process prog nil nil nil opts (expand-file-name tarfile dir))
	(message (format "Executing %s for %s ... done" prog tarfile))
	(mew-match 1 tarfile))
    tarfile
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Misc
;;;

(defun mew-summary-save (&optional user-coding-system)
  (interactive
   (list
    (and mew-mule-p current-prefix-arg
	 (read-coding-system "Coding-system: "))))
  (let ((msg (mew-summary-message-number))
	(part (mew-summary-part-number))
	(action "Save")
	(append-p nil)
	(cbuf (mew-summary-folder-name))
	file)
    (if (not (or msg part))
	(message "No message or part here")
      (setq file (mew-input-filename))
      (if (file-exists-p file)
	  (if (null mew-file-append-p)
	      (setq action "Overwrite")
	    (setq action "Append")
	    (setq append-p t)))
      (cond
       (part
	(if (mew-y-or-n-p (format "%s part %s to %s? " action part file))
	    (save-excursion
	      (set-buffer (mew-current-get 'cache))
	      (let* ((case-fold-search t)
		     (syntax (mew-syntax-get-entry-strnum
			      (mew-cache-decode-syntax
			       (mew-current-get 'cache)) part))
		     (begin (mew-syntax-get-begin syntax))
		     (end (mew-syntax-get-end syntax))
		     (ct (car (mew-syntax-get-ct syntax))))
		(if (string-match mew-ct-msg ct)
		    (setq end (mew-syntax-get-end
			       (mew-syntax-get-part syntax)))) ;;; xxxx
		(if (mew-member-match ct mew-mime-content-type-binary-list)
		    (mew-flet
		     (write-region begin end file append-p))
		  (mew-frwlet
		   mew-cs-noconv mew-cs-outfile
		   (write-region begin end file append-p)
		   ))
		)
	      (message "Wrote to %s" file)
	      ))
	)
       (msg
	(if (mew-y-or-n-p (format "%s message %s to %s? " action msg file))
	    (save-excursion
	      (set-buffer (get-buffer-create mew-buffer-tmp))
	      (widen)
	      (erase-buffer)
	      (mew-flet
	       (insert-file-contents (mew-expand-file-name msg cbuf))
	       (if (and mew-mule-p user-coding-system (not append-p))
		   (progn
		     (mew-cs-decode-region (point-min) (point-max) 
					   mew-cs-autoconv)
		     (mew-cs-encode-region (point-min) (point-max) 
					   user-coding-system)))
	       (write-region (point-min) (point-max) file append-p nil nil))
	      (message "Wrote to %s" file)
	      )
	  )
	)
       ))
    ))
      
(defun mew-summary-print ()
  (interactive)
  (mew-summary-display)
  (save-excursion
    (set-buffer (mew-buffer-message))
    (save-restriction
      (widen)
      (if (mew-y-or-n-p "Print this message? ")
	  (funcall mew-print-function))
      )
    ))

(defun mew-summary-jump-message (&optional msg)
;; if msg is t, jump last
  (interactive)
  (let ((here (point)))
    (if (null msg) (setq msg (read-string "Message No. : " "")))
    (cond 
     ((equal msg "") ())
     ((equal msg t)
      (goto-char (point-max))) ;; (forward-line -1)
     (t 
      (goto-char (point-min))
      (if (re-search-forward (format "^[ ]*%s[^0-9]+" msg) nil t) ;; xxx regex?
	  (beginning-of-line)
	(goto-char here))))
    ))

(defun mew-summary-jump-to-draft-buffer ()
  (interactive)
  (let ((num)
	(x (buffer-list)))
    (while x
      (if (string-match "^\\+drafts/" (buffer-name (car x)))
	  (setq num
		(cons (substring (buffer-name (car x)) (match-end 0)) num)))
      (setq x (cdr x)))
    (cond ((null num)
	   (error "No drafts buffer exist!"))
	  (t
	   (switch-to-buffer
	    (concat "+drafts/"
		     (car (sort num
				(function (lambda(a b)
					    (not (string< a b))))))))))))

(defun mew-summary-convert-local-charset ()
  (interactive)
  (if mew-mule-p
      (save-excursion
	(set-buffer (mew-buffer-message))
	(goto-char (point-min))
	(if (eq mew-message-citation 'header)
	    (progn
	      (re-search-forward mew-eoh nil t)
	      (forward-line)
	      ))
	(let ((buffer-read-only nil))
	  (mew-cs-decode-region (point) (point-max) mew-cs-rfc822-trans)
	  )
	))
  )

(defun mew-summary-redist ()
  (interactive)
  (cond 
   ((mew-summary-message-number)
    (mew-current-set 'window (current-window-configuration))
    (let* ((file (mew-expand-file-name (mew-summary-message-number)
				       (mew-summary-folder-name)))
	   (to (mew-input-address "Resent-To:"))
	   (cc (if mew-ask-cc (mew-input-address "Resent-Cc:")))
	   (draft (mew-expand-file-name (mew-draft-get-new) mew-draft-folder)))
      (save-excursion
	(set-buffer (find-file-noselect draft))
	(if mew-redist-needs-full-contents
	    (progn
	      (insert-file-contents file)
	      (mew-header-resent-lines mew-field-resent)
	      (mew-header-delete-lines mew-field-delete)
	      (goto-char (point-min))))
	(if (or (null to) (equal to ""))
	    ()
	  (mew-header-insert-here "Resent-To:" to))
	(if (or (null cc) (equal cc ""))
	    ()
	  (mew-header-insert-here "Resent-Cc:" cc))
	(save-buffer) 
	(message "Redistributing ... ")
	(if mew-redist-needs-full-contents
	    (call-process mew-prog-shell nil nil nil mew-prog-shell-arg
			  (format mew-redist-format1
				  mew-prog-send (buffer-file-name)))
	  (call-process mew-prog-shell nil nil nil mew-prog-shell-arg
			(format mew-redist-format2
				file mew-prog-send (buffer-file-name))))
	(message "Redistributing ... done")
	(kill-buffer (current-buffer))
	))
    )
   (t (message "No message"))
   )
  )

(defun mew-summary-isearch-forward ()
  (interactive)
  (let ((cwin (get-buffer-window (current-buffer)))
	(mwin (get-buffer-window (mew-buffer-message))))
    (if mwin 
	(progn
	  (select-window mwin)
	  (unwind-protect
	      (isearch-forward)
	    (select-window cwin))
	  ))
    ))

(defun mew-summary-isearch-backward ()
  (interactive)
  (let ((cwin (get-buffer-window (current-buffer)))
	(mwin (get-buffer-window (mew-buffer-message))))
    (if mwin 
	(progn
	  (select-window mwin)
	  (unwind-protect
	      (isearch-backward)
	    (select-window cwin))
	  ))
    ))

(defun mew-summary-x-face ()
  (interactive)
  (save-excursion
    (let (xface)
      (set-buffer (mew-buffer-message))
      (if (null (setq xface (mew-header-get-value "X-Face:")))
	  ()
	(set-buffer mew-buffer-tmp)
	(widen)
	(erase-buffer)
	(insert xface)
	(let ((filters mew-x-face-filter) file)
	  (while filters
	    ;; call-process-region is OK...
	    (call-process-region (point-min) (point-max) 
				 (car filters)
				 'delete t nil)
	    (setq filters (cdr filters))
	    )
	  (if (not (file-exists-p mew-temp-dir))
	      (mew-make-directory mew-temp-dir)) ;; just in case
	  (setq file (make-temp-name mew-temp-file))
	  ;; NEVER use call-process-region for privary reasons
	  (mew-flet
	   (write-region (point-min) (point-max) file))
	  (mew-mime-start-process mew-x-face-prog mew-x-face-args file)
	  )
	))
    ))

(defun mew-summary-pipe-message (prefix command)
  (interactive
   (list current-prefix-arg 
	 (read-string "Shell command on message: " mew-last-shell-command)))
  (mew-summary-display)
  (save-excursion
    (set-buffer (mew-buffer-message))
    (widen)
    (if (string-equal command "")
	(setq command mew-last-shell-command))
    (goto-char (point-min)) ; perhaps this line won't be necessary
    (if prefix
	(search-forward "\n\n"))
    (shell-command-on-region (point) (point-max) command nil)
    (setq mew-last-shell-command command))
  (mew-message-narrow-to-page)
  )

;;
;; Burst; This is not perfect, though...
;;

(defun mew-summary-burst ()
  (interactive)
  (save-excursion
    (if (null (mew-summary-message-number))
	(message "No message")
      (mew-flet
       (let ((target (mew-expand-file-name (mew-summary-message-number)
					   (mew-summary-folder-name)))
	     (n 1) (m 1) syntax len entry dir multi)
	 (message "Bursting ... ")
	 (set-buffer mew-buffer-tmp)
	 (widen)
	 (erase-buffer)
	 (insert-file-contents target)
	 (setq syntax (mew-decode (mew-decode-syntax-rfc822-head) t))
	 (setq multi (mew-syntax-get-part syntax))
	 (if (not (mew-syntax-multipart-p multi))
	     (message "Can't burst")
	   (setq dir (mew-input-directory-name))
	   (setq len (- (length multi) mew-syntax-magic))
	   (while (<= n len)
	     (setq entry (mew-syntax-get-entry syntax (list n)))
	     (if (not (equal (downcase (car (mew-syntax-get-ct entry)))
			     (downcase mew-ct-msg)))
		 ()
	       (write-region
		(mew-syntax-get-begin entry)
		(mew-syntax-get-end entry)
		(expand-file-name (int-to-string m) dir))
	       (setq m (1+ m)))
	     (setq n (1+ n))
	     )
	   (message "Bursting ... done")
	   )
	 ))
      )
    ))

(defun mew-summary-kill-subprocess ()
  (interactive)
  (if (not (processp mew-summary-buffer-process))
      (message "No process to kill")
    (message "%s was killed" (process-name mew-summary-buffer-process))
    (kill-process mew-summary-buffer-process)
    (setq mew-summary-buffer-process nil)
    )
  )

(defun mew-status-update (arg)
  (interactive "P")
  (cond
   (arg ;; C-u
    (message "Updating aliases and folders ... ")
    (setq mew-alias-alist (mew-alias-make-alist))
    (setq mew-folder-list (mew-folder-make-list))
    (setq mew-folder-alist (mew-folder-make-alist mew-folder-list))
    (setq mew-refile-alist (mew-refile-make-alist mew-folder-list))
    (message "Updating aliases and folders ... done")
    )
   (t
    (message "Updating aliases ... ")
    (setq mew-alias-alist (mew-alias-make-alist))
    (setq mew-refile-alist (mew-refile-make-alist mew-folder-list))
    (message "Updating aliases ... done"))
   )
  )

(provide 'mew-summary)
