;;;			     mew-pick.el
;;;
;;;	      Copyright (C) 1996-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: October  2, 1996
;;; Revised: April   21, 1997
;;;

(defconst mew-pick-version "mew-pick.el version 0.06")

(require 'mew)

(defvar mew-pick-map nil)

(if mew-pick-map
    ()
  (setq mew-pick-map (make-sparse-keymap))
  (define-key mew-pick-map "\t"   'mew-pick-complete)
  (define-key mew-pick-map "\r"   'exit-minibuffer)
  (define-key mew-pick-map "\n"   'exit-minibuffer)
  (define-key mew-pick-map "\C-g" 'abort-recursive-edit)
  )

(defvar mew-prog-pick-default-arg "-from")

(defvar mew-prog-pick-argalist
  '(("-from") ("-date") ("-cc") ("-to") ("-search") ("-before")
    ("-after") ("-subject") ("-not") ("-or") ("-and")
    ("-lbrace") ("-rbrace")))

(defvar mew-prog-pick-arglogic-unary
  '("-not"))

(defvar mew-prog-pick-arglogic-binary
  '("-or" "-and"))

(defvar mew-prog-pick-argdate
  '("-before" "-after"))

(defvar mew-prog-pick-argdatealist
  '(("yesterday") ("today") ("tomorrow")
    ("sunday") ("monday") ("tuesday") ("wednesday")
    ("thursday") ("friday") ("saturday")))

(defvar mew-pick-macro-alist nil
  "mew-pick-define-macro uses this variable.")

(defvar mew-prog-scan-default-range "update"
  "*A value to decide default range of scan.")

(defvar mew-prog-scan-default-range-alist '(("+drafts" . "all"))
  "*alist of folder and default range pair for scan."
  )


;;
;; Pick
;;

(defun mew-pick-define-macro (str1 str2)
  (interactive (list
		(concat "-" (read-string "Pick pattern: -"))
		(read-string "Macro Body: ")))
  (let* ((list (mew-header-split str1 ?\ ))
	 (assoc (assoc (car list) mew-pick-macro-alist))
	 (body (mew-header-split str2 ?\ ))
	 (info (cons (cdr list) body)))
    (if assoc
	(setcdr assoc info)
      (setq mew-pick-macro-alist
	    (cons (cons (car list) info)
		  mew-pick-macro-alist)))))

(defun mew-pick-macro-expand (patlist)
  (cond ((null patlist) nil)
	((assoc (car patlist) mew-prog-pick-argalist)
	 (let ((first (car patlist))
	       (rest (cdr patlist)))
	   (if (or (mew-member first mew-prog-pick-arglogic-binary)
		   (mew-member first mew-prog-pick-arglogic-unary)
		   (mew-member first '("-lbrace" "-rbrace")))
	       (cons first
		     (mew-pick-macro-expand rest))
	     (cons first
		   (cons (car rest)
			 (mew-pick-macro-expand (cdr rest)))))))
	((mew-member (car patlist) mew-prog-pick-argdate)
	 (cons (car patlist)
	       (cons (nth 1 patlist)
		     (mew-pick-macro-expand (nthcdr 2 patlist)))))
	(t
	 (let ((assoc (assoc (car patlist)
			     mew-pick-macro-alist)))
	   (if (not assoc)
	       (cons (car patlist)
		     (mew-pick-macro-expand (cdr patlist)))
	     (let ((args (nth 1 assoc))
		   (body (copy-sequence (nthcdr 2 assoc)))
		   arg find replc (replclist (cdr patlist)))
	       (while (and args replclist)
		 (setq arg (car args))
		 (setq args (cdr args))
		 (setq replc (car replclist))
		 (setq replclist (cdr replclist))
		 (setq find (member arg body))
		 (while find 
		   (setcar find replc)
		   (setq find
			 (member arg body))))
	       (mew-pick-macro-expand
		(append body replclist))))))))

(defun mew-pick-complete ()
  (interactive)
  (let ((word nil) (comp nil) (all nil)
	(alist (append mew-pick-macro-alist
		       mew-prog-pick-argalist)))
    (if (null (setq word (mew-delete-backward-char)))
	()
      (setq comp (try-completion word alist))
      (setq all (all-completions word alist))
      (cond
       ((eq comp t)
	(insert (car (assoc word alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)))
       ((stringp comp)
	(insert comp)
	(and (> (length all) 1)
	     (with-output-to-temp-buffer mew-buffer-completions
	       (display-completion-list all))))
       (t (insert word)
	  (ding))
       )
      )
    ))

(defun mew-pick-input-pattern (&optional prompt)
  (mew-pick-macro-expand
   (mew-pick-input-pattern-raw prompt)))

(defun mew-pick-input-pattern-raw (&optional prompt)
  (let ((pat nil))
    (setq pat (read-from-minibuffer 
	       (format "Pattern%s (%s) : "
		       (if prompt prompt "")
		       mew-prog-pick-default-arg)
	       "-" mew-pick-map nil))
    (setq pat (downcase pat))
    (cond
     ((string-match " " pat)
      (mapcar (function mew-summary-search-blace)
	      (mew-header-delete-nullstring-list
	       (mew-header-split pat 32))));; 32 is " "
     (t 
      (if (or (string= pat "") (string= pat "-"))
	  (setq pat mew-prog-pick-default-arg))
      (cond
       ((mew-member pat mew-prog-pick-arglogic-unary)
	(cons pat (mew-pick-input-pattern-raw (format " for %s" pat))))
       ((mew-member pat mew-prog-pick-arglogic-binary)
	(mew-pick-input-logic pat))
       ((mew-member pat mew-prog-pick-argdate)
	(cons pat (mew-pick-input-value (format "Date for %s" pat)
					mew-prog-pick-argdatealist)))
       ((assoc pat mew-pick-macro-alist)
	(let* ((assoc (assoc pat mew-pick-macro-alist))
	       (args (nth 1 assoc)) arg n datep res)
	  (setq res (list pat))
	  (while args
	    (setq arg (car args))
	    (setq args (cdr args))
	    (setq n (and (string-match "[0-9]+" arg) (mew-match 0 arg)))
	    (setq datep (string-match "d" arg))
	    (setq res
		  (nconc
		   (mew-pick-input-value 
		    (cond ((and n datep)
			   (format "Date arg %s for %s" n pat))
			  (n
			   (format "Arg %s for %s" n pat))
			  (datep
			   (format "Date for %s" pat))
			  (t
			   (format "Value for %s" pat)))
		    (and datep mew-prog-pick-argdatealist))
		   res)))
	  (nreverse res)))
       (t (cons pat (mew-pick-input-value (format "Value for %s" pat))))
       )
      )
    )))

;; pattern = key value | logic

(defun mew-pick-input-value (prompt &optional alist)
  (let ((ret nil))
    (cond
     (alist (setq ret (completing-read (format "%s : " prompt)
				       alist
				       nil
				       nil  ;; not require match
				       "")))
     (t (setq ret (read-string (format "%s : " prompt) "")))
     )
    (list ret)
    ))

(defun mew-pick-input-logic (logic)
  (append (mew-pick-input-pattern-raw (format "1 for %s" logic))
	  (cons logic 
		(mew-pick-input-pattern-raw (format "2 for %s" logic))))
  )

(defun mew-summary-search-blace (string)
  (cond
   ((equal string "(") "-lbrace")
   ((equal string ")") "-rbrace")
   (t string)))

(defun mew-summary-search ()
  (interactive)
  (let ((folder (mew-input-folder (buffer-name)))
	(pattern nil)
	(range nil))
    (if (null (file-directory-p (mew-expand-file-name folder)))
	(message "No such folder %s" folder)
      (setq pattern (mew-pick-input-pattern))
      (message "Picking messages in %s ..." folder)
      (setq range (mew-summary-pick folder pattern))
      (message "Picking messages in %s ... done" folder)
      (if (get-buffer folder)
	  (switch-to-buffer folder)
	(mew-summary-folder-create folder))
      (if range (mew-summary-scan-body folder (list range 'erase)))
      )
    ))

(defun mew-summary-search-mark ()
  (interactive)
  (if (equal (point-min) (point-max))
      (message "No messages in this buffer.")
    (let ((folder (buffer-name))
	  (pattern nil)
	  (first nil)
	  (last nil)
	  (range nil))
      (setq pattern (mew-pick-input-pattern))
      (message "Picking messages in %s ..." folder)
      (goto-char (point-min))
      (setq first (mew-summary-message-number))
      (goto-char (point-max))
      (forward-line -1)
      (setq last (mew-summary-message-number))
      (setq range (mew-summary-pick folder pattern (concat first "-" last)))
      (message "Picking messages in %s ... done" folder)
      (if (null range)
	  ()
	(message "Marking messages ... ")
	(goto-char (point-min))
	(while (not (eobp))
	  (if (and (null (mew-summary-marked-p))
		   (mew-member-del (mew-summary-message-number) range))
	      (mew-summary-mark mew-mark-hop))
	  (forward-line))
	(message "Marking messages ... done")
	)
      )))

(defun mew-member-del (a list)
  (let ((pointer (cons nil list)))
    (if (stringp a)
	(let ((b (concat a "/")))
	  (catch 'member
	    (while list
	      (if (or (equal (car list) a)
		      (equal (car list) b))
		  (progn
		    (setcdr pointer (cdr list))
		    (throw 'member t))
		(setq list (cdr list))
		(setq pointer (cdr pointer))
		))))
      (catch 'member
	(while list
	  (if (equal (car list) a)
	      (progn
		(setcdr pointer (cdr list))
		(throw 'member t))
	    (setq list (cdr list))
	    (setq pointer (cdr pointer))
	    )))))
  )

(defun mew-summary-pick (folder pattern &optional range)
  (let ((epat (mapcar
	       (function (lambda (x) (mew-cs-encode-string x mew-cs-scan)))
	       pattern))
	start msgs)
    (setq range (or range "all"))
    (save-excursion
      (set-buffer (get-buffer-create mew-buffer-tmp))
      (erase-buffer)
      ;; input(result) from pick is line-based stream...
      (mew-piolet
       mew-cs-autoconv mew-cs-noconv
       (apply (function call-process)
	      mew-prog-pick nil t nil "-list" folder range epat)
       )
      (goto-char (point-min))
      (cond 
       ((looking-at "pick: no messages") (message "No such messages") nil)
       ((looking-at "pick: ") (message "Illegal pattern") nil)
       (t 
	(while (not (eobp))
	  (setq start (point))
	  (forward-line)
	  (setq msgs (cons (mew-buffer-substring start (1- (point))) msgs))
	  )
	(nreverse msgs))
       )
      )))

(provide 'mew-pick)
