;;;			      mew-pgp.el
;;;
;;;	      Copyright (C) 1994-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: August 17, 1994
;;; Revised: April  21, 1997
;;;

(defconst mew-pgp-version "mew-pgp.el version 0.38")

(require 'mew)

(defvar mew-prog-pgp "pgp")
(defvar mew-pgp-key-suffix ".pka")
(defvar mew-pgp-ascii-suffix ".asc")

(setq mew-pgp-string nil)
(setq mew-pgp-running nil)

(setq mew-pgp-sign-failure nil)

(setq mew-pgp-decrypt-failure nil)
(setq mew-pgp-decrypt-msg nil)

(defconst mew-pgp-key-begin "-----BEGIN PGP PUBLIC KEY BLOCK-----")
(defconst mew-pgp-key-end   "-----END PGP PUBLIC KEY BLOCK-----")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; a password function
;;

(defun mew-read-passwd (prompt)
  (let ((pass)
	(c 0)
	(echo-keystrokes 0)
	(ociea cursor-in-echo-area))
    (unwind-protect
	(progn
	  (setq cursor-in-echo-area 1)
	  (while (and (/= c ?\r) (/= c ?\n) (/= c ?\e) (/= c 7)) ;; ^G
	    (message "%s%s"
		     prompt
		     (make-string (length pass) ?.))
	    (setq c (read-char))
	    (if (char-equal c ?\C-u)
		(setq pass "")
	      (if (and (/= c ?\b) (/= c ?\177))
		  (setq pass (concat pass (char-to-string c)))
		(if (> (length pass) 0)
		    (setq pass (substring pass 0 -1))))))
	  (setq cursor-in-echo-area -1)
	  )
      (setq cursor-in-echo-area ociea)
      nil)
    (message "")
    (sit-for 0)
    (substring pass 0 -1)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; PGP encrypting
;;

(defun mew-pgp-encrypt (file1 decrypters)
  (message "PGP encrypting ... ")
  (let (file2 file3) ;; not unique if makes temp here
    (if (not (file-exists-p mew-temp-dir))
	(mew-make-directory mew-temp-dir)) ;; just in case
    (setq file2 (make-temp-name mew-temp-file)) 
    (save-excursion
      (set-buffer mew-buffer-tmp)
      (widen)
      (erase-buffer)
      (insert "Version: 1\n")
      (write-region (point-min) (point-max) file2))
    (setq file3 (concat (make-temp-name mew-temp-file) mew-pgp-ascii-suffix))
    (apply (function call-process) 
	   mew-prog-pgp nil nil nil
	   "-ea" "+language=en" "+batchmode=on" "+armorlines=0"
	   "-o" file3 file1 decrypters)
    (message "PGP encrypting ... done")
    (list file2 nil file3 nil);; both ctes are 7bit
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; PGP decrypting
;;

(defun mew-pgp-decrypt (file1 file2)
  ;; file1 is a key file. just ignore.
  ;; file2 is an encrypted file with PGP.
  (message "PGP decrypting ... ")
  (setq mew-pgp-running t)
  (setq mew-pgp-string nil)
  (setq mew-pgp-decrypt-msg nil)
  (setq mew-pgp-decrypt-failure nil)
  (let ((process-connection-type mew-connection-type2)
	file3 process)
    (if (not (file-exists-p mew-temp-dir))
	(mew-make-directory mew-temp-dir)) ;; just in case
    (setq file3 (make-temp-name mew-temp-file)) 
    (setq process (start-process "PGP decrypt"
				 (current-buffer) ;; xxx
				 mew-prog-pgp
				 "+language=en" "+batchmode=off"
				 "-o" file3 file2))
    (mew-set-process-cs process mew-cs-noconv mew-cs-noconv)
    (set-process-filter process 'mew-pgp-decrypt-filter1)
    (set-process-sentinel process 'mew-pgp-decrypt-sentinel)
    ;; Wait for the termination of PGP.
    ;; Emacs doesn't provide synchronize mechanism with
    ;; an asynchronous process. So, take this way. 
    (if mew-xemacs-p
	(while mew-pgp-running
	  (accept-process-output))
      (while mew-pgp-running
	(sit-for 1)
	;; accept-process-output or sleep-for is not enough
	(discard-input)))
    (message "PGP decrypting ... done")
    (list file3 mew-pgp-decrypt-msg)
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; PGP signing
;;

(defun mew-pgp-sign (file1)
  (message "PGP signing ... ")
  (setq mew-pgp-running t)
  (setq mew-pgp-string nil)
  (setq mew-pgp-sign-failure nil)
  (let ((process-connection-type mew-connection-type2)
	file2 process)
    (if (not (file-exists-p mew-temp-dir))
	(mew-make-directory mew-temp-dir)) ;; just in case
    (setq file2 (concat (make-temp-name mew-temp-file) mew-pgp-ascii-suffix))
    ;; not perfectly unique but OK
    (setq process (start-process "PGP sign"
				 (current-buffer) ;; xxx
				 mew-prog-pgp
				 "+language=en" "+batchmode=off"
				 "-o" file2
				 "-sab" file1))
    (set-process-filter process 'mew-pgp-sign-filter1)
    (set-process-sentinel process 'mew-pgp-sign-sentinel)
    ;; Wait for the termination of PGP.
    ;; Emacs doesn't provide synchronize mechanism with
    ;; an asynchronous process. So, take this way. 
    (if mew-xemacs-p
	(while mew-pgp-running
	  (accept-process-output))
      (while mew-pgp-running
	(sit-for 1)
	;; accept-process-output or sleep-for is not enough
	(discard-input)))
    (message "PGP signing ... done")
    (if mew-pgp-sign-failure
	(progn
	  (setq mew-pgp-sign-failure nil)
	  (list nil nil nil)) ;; return
      (list file2 nil "pgp-md5")) ;; return
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; PGP verifying
;;

(defun mew-pgp-verify (file1 file2)
  (message "PGP verifying ... ")
  (let (ret)
    (save-excursion
      (set-buffer mew-buffer-tmp)
      (widen)
      (erase-buffer)
      (call-process mew-prog-pgp nil t nil 
		    "+batchmode=on" "+language=en" file2 file1)
      (message "PGP verifying ... done")
      (goto-char (point-min))
      (if (not (re-search-forward 
		"\n\\(.*\\) signature from user \\(.*\\)\\." nil t))
	  (if (re-search-forward "Can't find the right public key" nil t)
	      (setq ret (concat ret "No public key."))
	    (goto-char (point-min))
	    (if (re-search-forward  "Keyring file.* does not exist" nil t)
		(setq ret (concat ret "No keyring."))
	      (goto-char (point-min))))
	(setq ret (concat (mew-match 1) " PGP sign " (mew-match 2)))
	(if (re-search-forward "not certified with enough" nil t)
	    (setq ret (concat ret " MARGINAL"))
	  (goto-char (point-min))
	  (if (re-search-forward "not trusted" nil t)
	      (setq ret (concat ret " UNTRUSTED"))
	    (goto-char (point-min))
	    (if (re-search-forward "not certified with a" nil t)
		;; PGP uses "unknown" for validity internally, but
		;; prints "undefined" instead of "unknown".
		(setq ret (concat ret " UNDEFINED"))
	      (setq ret (concat ret " COMPLETE"))))))
      )
    ret
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; PGP decrypting process funtions
;;

(defun mew-pgp-decrypt-sentinel (process event)
  (if mew-pgp-decrypt-failure
      (cond
       ((eq mew-pgp-decrypt-failure 'nofile)
	(setq mew-pgp-decrypt-msg "The PGP keyring doesn't exist."))
       ((eq mew-pgp-decrypt-failure 'nopass)
	(setq mew-pgp-decrypt-msg "Pass phrases are wrong."))
       ((eq mew-pgp-decrypt-failure 'nouserid)
	(setq mew-pgp-decrypt-msg "The secret key doesn't exist."))
       (t
	(setq mew-pgp-decrypt-msg "Decrypting failed sor some reasons."))
       )
    (setq mew-pgp-decrypt-msg "PGP decrypted."))
  (setq mew-pgp-running nil)
  (setq mew-pgp-string nil)
  )

(defun mew-pgp-decrypt-filter1 (process string)
  (setq mew-pgp-string (concat mew-pgp-string string))
  (cond
   ((string-match "Enter pass phrase: $" mew-pgp-string)
    (set-process-filter process 'mew-pgp-decrypt-filter2)
    (process-send-string
     process
     (format "%s\n" (mew-read-passwd "Enter pass phrase : ")))
    )
   ((string-match "You do not have the secret key" mew-pgp-string)
    (setq mew-pgp-decrypt-failure 'nouserid)
    (set-process-filter process 'mew-pgp-decrypt-filter3))
   )
  )

(defun mew-pgp-decrypt-filter2 (process string)
  (setq mew-pgp-string (concat mew-pgp-string string))
  (cond
   ((string-match "No passphrase" mew-pgp-string)
    ;; pass phrases are wrong three times
    (setq mew-pgp-decrypt-failure 'nopass)
    (set-process-filter process 'mew-pgp-decrypt-filter3))
   ((string-match "Error:  Bad pass phrase.\n\nEnter pass phrase: $"
		  mew-pgp-string)
    (setq mew-pgp-string nil)
    (process-send-string 
     process
     (format "%s\n" (mew-read-passwd "Re-enter pass phrase : ")))
    ;; never change filter
    )
   )
  )

(defun mew-pgp-decrypt-filter3 (process string)
  () ;; do nothing
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; PGP signing process funtions
;;

(defun mew-pgp-sign-sentinel (process event)
  (set-buffer (process-buffer process))
  (if mew-pgp-sign-failure
      (cond
       ((eq mew-pgp-sign-failure 'nofile)
	(message "secring doesn't exist."))
       ((eq mew-pgp-sign-failure 'nouserid)
	(message "userid %s is not found in secring." (user-login-name)))
       ((eq mew-pgp-sign-failure 'nopass)
	(message "Pass phrase is wrong"))
       (t (message "Signing failed for some reasons"))
       ))
  (setq mew-pgp-running nil)
  (setq mew-pgp-string nil)
  )

(defun mew-pgp-sign-filter1 (process string)
  (setq mew-pgp-string (concat mew-pgp-string string))
  (cond
   ((string-match "Enter pass phrase: $" mew-pgp-string)
    (set-process-filter process 'mew-pgp-sign-filter2)
    (process-send-string
     process
     (format "%s\n" (mew-read-passwd "Enter pass phrase : ")))
    )
   ((string-match "Keyring file" mew-pgp-string) ;; no secring
    (setq mew-pgp-sign-failure 'nofile)
    (set-process-filter process 'mew-pgp-sign-filter3))
   ((string-match "Key matching userid" mew-pgp-string) ;; no userid in secring
    (setq mew-pgp-sign-failure 'nouserid)
    (set-process-filter process 'mew-pgp-sign-filter3))
   (t ()) ;; just in case
   )
  )

(defun mew-pgp-sign-filter2 (process string)
  (setq mew-pgp-string (concat mew-pgp-string string))
  (cond
   ((string-match "Pass phrase is good." mew-pgp-string)
    (message "PGP signing ... ")
    (set-process-filter process 'mew-pgp-sign-filter3))
   ((string-match "No passphrase" mew-pgp-string)
    ;; pass phrases are wrong three times
    (setq mew-pgp-sign-failure 'nopass)
    (set-process-filter process 'mew-pgp-sign-filter3))
   ((string-match "Error:  Bad pass phrase.\n\nEnter pass phrase: $"
		  mew-pgp-string)
    (setq mew-pgp-string nil)
    (process-send-string 
     process
     (format "%s\n" (mew-read-passwd "Re-enter pass phrase : ")))
    ;; never change filter
    )
   )
  )

(defun mew-pgp-sign-filter3 (process string)
  () ;; just ignore
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; shortcut methods
;;

(defun mew-pgp-sign-letter ()
  "Sign the single part draft with PGP. Input your passphrase."
  (interactive)
  (mew-pgp-encode-letter (list (list mew-ct-mls mew-ct-pgs)))
  )

(defun mew-pgp-encrypt-letter ()
  "Encrypt the single part draft with PGP."
  (interactive)
  (mew-pgp-encode-letter (list (list mew-ct-mle mew-ct-pge)))
  )

(defun mew-pgp-sign-encrypt-letter ()
  "Sign then encrypt the single part draft with PGP. Input your passphrase."
  (interactive)
  (mew-pgp-encode-letter (list (list mew-ct-mls mew-ct-pgs)
			       (list mew-ct-mle mew-ct-pge)))
  )

(defun mew-pgp-encode-letter (privacy)
  (cond
   ((null (mew-which mew-prog-pgp exec-path))
    (message "PGP doesn't exist"))
   ((mew-header-get-value "Content-Type:")
    (message "You can't sign/encrypt this draft due to Content-Type:"))
   ((mew-attach-p)
    (message "You can't sign/encrypt this draft with the shortcut method"))
   (t
    (goto-char (marker-position mew-draft-buffer-header))
    ;; the beginning of "----"
    (let ((beg (point)) decrypters)
      (forward-line)
      ;; cursor is just after "----\n"
      (delete-region beg (point))
      ;; same as mew-encode-decrypters-list
      (setq decrypters (mew-header-decrypters-collect '("To:" "Cc:")))
      (mew-encode-singlepart
       (mew-encode-syntax-single "text-file" nil nil nil privacy decrypters)
       nil nil t)
      (goto-char (point-min))
      (sit-for 0) ;; to display the top of buffer 
      (re-search-forward mew-eoh2)
      (beginning-of-line)
      (setq mew-draft-buffer-header (point-marker)) ;; just in case
      ))
   ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; key distribution
;;

(defun mew-attach-pgp-public-key ()
  "Extract the PGP key for the inputed user on \".\" in attachments"
  (interactive)
  (if (not (mew-attach-not-line012-1))
      (message "Can't link here.")
    (let* ((error nil)
	   (nums (mew-attach-nums))
	   (subdir (mew-attach-expand-path mew-encode-syntax nums))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (file (concat (mew-attach-random-filename) mew-pgp-key-suffix))
	   (cnt 0) (max (length mew-attach-random-filename))
	   user filepath begin end)
      ;; mimedir / {subdir/} dir
      (if (not (equal subdir "")) 
	  (setq mimedir (expand-file-name subdir mimedir)))
      ;; mimedir / file
      (setq filepath (expand-file-name file mimedir))
      (while (and (file-exists-p filepath) (< cnt max))
	(setq file (concat (mew-attach-random-filename) mew-pgp-key-suffix))
	(setq filepath (expand-file-name file mimedir))
	(setq cnt (1+ cnt)))
      (if (file-exists-p filepath)
	  (message "Could not make a file for pgp key, sorry.")
	(setq user (mew-input-address2 "Who's key? (%s): " mew-mail-address))
	(save-excursion
	  (set-buffer mew-buffer-tmp)
	  (widen)
	  (erase-buffer)
	  (call-process mew-prog-pgp nil t nil "-fkxa" user)
	  (goto-char (point-min))
	  (if (not (re-search-forward mew-pgp-key-begin nil t))
	      (setq error t)
	    (beginning-of-line)
	    (setq begin (point))
	    (if (not (re-search-forward mew-pgp-key-end nil t))
		(setq error t)
	      (beginning-of-line)
	      (forward-line)
	      (setq end (point)))
	    (write-region begin end filepath)))
	(if error
	    (message "can't extract pgp key for %s" user)
	  (setq mew-encode-syntax
		(mew-syntax-insert-entry
		 mew-encode-syntax 
		 nums
		 (mew-encode-syntax-single file mew-type-apk nil user)))
	  (mew-encode-syntax-print mew-encode-syntax))
	))
    ))
      
(defvar mew-pgp-tmp-file nil)

(defun mew-mime-pgp-keys (begin end &optional params)
  "A function to add PGP keys in Application/PGP-Keys to your 
public keyring."
  (interactive)
  (if (not (mew-which mew-prog-pgp exec-path))
      (message "PGP is not found")
    (if (not (mew-y-or-n-p "Add this PGP key onto your public keyring? "))
	()
      (if (not (file-exists-p mew-temp-dir))
	  (mew-make-directory mew-temp-dir));; just in case
      (setq mew-pgp-tmp-file (make-temp-name mew-temp-file))
      (save-excursion
	(set-buffer (mew-current-get 'cache))
	(mew-frwlet
	 (write-region begin end mew-pgp-tmp-file)
	 (message "");; flush write-region message
	 )
	(set-buffer (mew-buffer-message))
	(let ((buffer-read-only nil))
	  (message "Adding PGP keys ... ")
	  (call-process mew-prog-pgp nil t nil 
			"+batchmode=on" "-ka" mew-pgp-tmp-file)
	  (message "Adding PGP keys ... done")
	  (insert "\n\n"
		  "**************** IMPORTANT NOTE ****************\n"
		  "When Mew adds PGP keys onto your public keyring,\n"
		  "it is careless about both TRUST and VALIDITY.\n"
		  "It is YOU who set these values. Please use\n"
		  "\"pgp -ke\" and \"pgp -ks\" to change them.\n"
		  "If you don't know what TRUST and VALIDITY is,\n"
		  "you should learn the web of trust system BEFORE\n"
		  "using PGP to protect your privacy.\n"
		  "**************** IMPORTANT NOTE ****************\n"
		  )
	  ))
      (if (file-exists-p mew-pgp-tmp-file)
	  (delete-file mew-pgp-tmp-file))
      )
    ))

(provide 'mew-pgp)
