;;;			    mew-minibuf.el
;;;
;;;		Copyright (C) 1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: March 23, 1997
;;; Revised: 
;;;

(defconst mew-minibuf-version "mew-minibuf.el version 0.01")

(require 'mew)

(defvar mew-minibuffer-map      nil)
(defvar mew-folder-map          nil)

(if mew-minibuffer-map
    ()
  (setq mew-minibuffer-map (make-sparse-keymap))
  (define-key mew-minibuffer-map "\t"     'mew-minibuffer-alias)
  (define-key mew-minibuffer-map "\C-c\t" 'mew-minibuffer-domain)
  (define-key mew-minibuffer-map "\r"     'exit-minibuffer)
  (define-key mew-minibuffer-map "\n"     'exit-minibuffer)
  (define-key mew-minibuffer-map "\C-g"   'abort-recursive-edit)
  (if mew-use-bbdb
      (define-key mew-minibuffer-map "\e\t"     'bbdb-complete-name))
  )

(if mew-folder-map
    ()
  (setq mew-folder-map (make-sparse-keymap))
  (define-key mew-folder-map " "    'mew-minibuffer-folders-word)
  (define-key mew-folder-map "\t"   'mew-minibuffer-folders)
  (define-key mew-folder-map "\r"   'exit-minibuffer)
  (define-key mew-folder-map "\n"   'exit-minibuffer)
  (define-key mew-folder-map "\C-g" 'abort-recursive-edit)
  (define-key mew-folder-map "\M-p" 'previous-history-element)
  (define-key mew-folder-map "\M-n" 'next-history-element)
  )

;;;
;;; Input method
;;;

(defun mew-input-folder (default)
  (let ((folder))
    (setq folder (completing-read (format "Folder name (%s): " default)
				  mew-folder-alist
				  nil
				  nil  ;; not require match
				  "+"))
    (directory-file-name 
     (if (or (string= folder "") (string= folder "+"))
 	 default
       folder))))

(defvar mew-input-folder-hist nil)

(defun mew-input-folders (default &optional string)
  (let ((folders))
    (setq folders
	  (read-from-minibuffer (format "Folder name (%s): " default)
				(or string "+")
				mew-folder-map
				nil
				mew-input-folder-hist))
    (if (or (string= folders "") (string= folders "+"))
	(setq folders default))
    (mapcar (function directory-file-name) 
	    (mew-header-split
	     (mew-header-syntax folders) ?,))
    ))

(defun mew-input-range (folder)
  "Return (range erase-update)."
  (mew-decode-syntax-delete)
  ;; for the case when parts are expanded in the bottom of the folder
  (let*
      ((pair (assoc folder mew-prog-scan-default-range-alist))
       (default (or (cdr pair) mew-prog-scan-default-range))
       (range (if mew-ask-range
		  (read-string (format "Range (%s): " default) "")
		"")))
    (if (string= range "")
	(setq range default))
    (cond
     ;; range is other than "update"
     ((not (string= range "update"))
      (list range 'erase)) ;; non-update, erase it
     ;; update
     ((get-buffer folder)
      (save-excursion
	(set-buffer folder)
	(goto-char (point-max))
	(if (bobp)
	    (list "all" 'update) ;; buffer is empty. no need to erase
	  (forward-line -1)
	  (list 
	   (concat
	    (int-to-string (1+ (string-to-int (mew-summary-message-number))))
	    "-" 
	    "last")
	   'update) ;; this is update!
	  )))
     ;; update but folder doesn't exist in Emacs. 
     (t (list "all" 'update)) ;; no need to erase
     )
    ))

(defun mew-input-address (prompt)
  (read-from-minibuffer (concat prompt " ") "" mew-minibuffer-map nil)
  )

(defun mew-input-address2 (prompt default)
  (let ((ret (read-from-minibuffer 
	      (format prompt default) "" mew-minibuffer-map nil)))
    (if (string= ret "") default ret)
    ))

(defun mew-input-filename (&optional prompt default)
  (let ((msg (or prompt "File: "))
	(file (concat "~/" default))
	(use-dialog-box nil))
    (expand-file-name (read-file-name msg file))
    ))

(defun mew-input-directory-name ()
  (let ((dir (expand-file-name (read-file-name "Directory : " nil "~" t))))
    (if (file-directory-p dir)
	dir
      (mew-input-directory-name)
      )
    ))

(defun mew-input-string (prompt subdir default)
  (let ((input (read-string (format prompt subdir default) "")))
    (if (string= input "") default input))
  )

(defun mew-input-type (prompt filename default type-list)
  (let ((completion-ignore-case t)
	(type))
    (setq type (completing-read
		(format prompt filename default)
		(mapcar (function (lambda (x) (cons x x)))
			type-list)
		nil
		t  ;; not require match
		""))
    (if (string= type "") default type)
    ))

;;;
;;; mini buffer
;;;

(defun mew-minibuffer-alias ()
  (interactive)
  (let ((word nil) (comp nil) (all nil) (len nil))
    (if (null (setq word (mew-delete-backward-char)))
	(tab-to-tab-stop) ;; default key binding
      (setq comp (try-completion word mew-alias-alist))
      (setq all (all-completions word mew-alias-alist))
      (setq len (length word))
      (cond
       ((eq comp t)
	(insert (cdr (assoc word mew-alias-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)
	     (mew-window-pop)))
       ((and (char-equal (aref word (1- len)) ?@)
	     (assoc (substring word 0 (1- len)) mew-alias-alist))
	(insert (cdr (assoc (substring word 0 (1- len)) mew-alias-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)
	     (mew-window-pop)))
       ((stringp comp)
	(insert comp)
	(and (> (length all) 1)
	     (mew-window-push)
	     (with-output-to-temp-buffer mew-buffer-completions
	       (display-completion-list all))))
       (t (insert word)
	  (mew-temp-minibuffer-message " No matching alias"))
       )
      )
    ))

(defun mew-minibuffer-domain ()
  (interactive)
  (let ((word (mew-delete-backward-char-at)))
;;;;    (if (null (setq word (mew-delete-backward-char-at)))
;;;;	(lisp-complete-symbol) ;; default key binding)
    ;; word is strings between @ and cursor
    (cond
     ((equal word nil) ;; @ doesn't exist.
      (if (null mew-mail-domain-list)
	  ()
	(insert "@")
	(with-output-to-temp-buffer mew-buffer-completions
	  (display-completion-list mew-mail-domain-list))))
     ((equal word t) ;; just after @
      (if mew-mail-domain-list
	  (with-output-to-temp-buffer mew-buffer-completions
	    (display-completion-list mew-mail-domain-list)))
      )
     (t
      (let ((comp nil) (all nil)
	    (dalist (mew-slide-pair mew-mail-domain-list)))
	(setq comp (try-completion word dalist))
	(setq all (all-completions word dalist))
	(cond
	 ;; already completed
	 ((eq comp t) 
	  (insert (cdr (assoc word dalist))) ;; next candidate
	  (and (get-buffer mew-buffer-completions)
	       (kill-buffer mew-buffer-completions))
	  )
	 ;; just one candidate
	 ((equal 1 (length all))
	  (insert (car all))
	  (and (get-buffer mew-buffer-completions)
	       (kill-buffer mew-buffer-completions))
	  (mew-temp-minibuffer-message " Sole completion")
	  )
	 ;; one or more candidate
	 ((stringp comp)
	  (insert comp)
	  (and (> (length all) 1)
	       (with-output-to-temp-buffer mew-buffer-completions
		 (display-completion-list all))))
	 ;; no candidate
	 (t (insert word)
	    (mew-temp-minibuffer-message " No matching domain"))
	 ))
      )
     )
    ))

;;
;; folder completion
;;

(defun mew-temp-minibuffer-message (m)
  (let ((savemax (point-max)))
    (save-excursion
      (goto-char (point-max))
      (insert m))
    (let ((inhibit-quit t))
      (sit-for 2)
      (delete-region savemax (point-max))
      (if quit-flag (setq quit-flag nil	unread-command-events 7))
      )))

(defun mew-minibuffer-folders ()
  (interactive)
  (let (word comp)
    (if (null (setq word (mew-delete-backward-char)))
	(tab-to-tab-stop) ;; default key binding
      (setq comp (try-completion word mew-folder-alist))
      (cond
       ((eq comp t) 
	;; insert completed word
	(insert (cdr (assoc word mew-folder-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)))
       ((stringp comp)
	(insert comp)
	(if (string-equal comp word)
	    (mew-temp-minibuffer-message " [Complete, but not unique]")))
       (t (insert word)
	  (mew-temp-minibuffer-message " No matching folder"))
       )
      )
    ))

(defun mew-minibuffer-folders-word ()
  (interactive)
  (let (word comp all)
    (if (null (setq word (mew-delete-backward-char)))
	(tab-to-tab-stop) ;; default key binding
      (setq comp (try-completion word mew-folder-alist))
      (setq all (all-completions word mew-folder-alist))
      (cond
       ((eq comp t) 
	;; insert completed word
	(insert (cdr (assoc word mew-folder-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)))
       ((stringp comp)
	(insert comp) ;; insert substring
	(and (> (length all) 1)
	     (with-output-to-temp-buffer mew-buffer-completions
	       (display-completion-list all))))
       (t (insert word)
	  (mew-temp-minibuffer-message " No matching folder"))
       )
      )
    ))

(provide 'mew-minibuf)
