;;;			    mew-message.el
;;;
;;;	      Copyright (C) 1996-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: October  2, 1996
;;; Revised: April   21, 1997
;;;

(defconst mew-message-version "mew-message.el version 0.05")

(require 'mew)

(defvar mew-message-mode-map    nil)

(defvar mew-message-mode-menu-spec
  '("Mew/message"
    ["Next part"    mew-message-next-msg t]
    ["Prev part"    mew-message-prev-msg t]
    ["Next page"    mew-message-next-page t]
    ["Prev page"    mew-message-prev-page t]
    ["Goto summary" mew-message-goto-summary t]
    "---"
    ("Reply/Forward"
     ["Reply"               mew-message-reply t]
     ["Reply with citation" mew-message-reply-with-citation t]
     ["Forward"             mew-message-forward t]
     ["Redistribute"        mew-message-redist t]
     )
    ["Delete" mew-message-rmm t]
    ["Refile" mew-message-refile t]
    )
  )

(if mew-message-mode-map
    ()
  (setq mew-message-mode-map (make-sparse-keymap))
  (define-key mew-message-mode-map " "    'mew-message-next-page)
  (define-key mew-message-mode-map "\177" 'mew-message-prev-page)
  (define-key mew-message-mode-map "n"    'mew-message-next-msg)
  (define-key mew-message-mode-map "p"    'mew-message-prev-msg)
  (define-key mew-message-mode-map "h"    'mew-message-goto-summary)
  (define-key mew-message-mode-map "a"    'mew-message-reply)
  (define-key mew-message-mode-map "A"    'mew-message-reply-with-citation)
  (define-key mew-message-mode-map "f"    'mew-message-forward)
  (define-key mew-message-mode-map "r"    'mew-message-redist)
  (define-key mew-message-mode-map "d"    'mew-message-rmm)
  (define-key mew-message-mode-map "o"    'mew-message-refile)
  (if mew-emacs-p
      (easy-menu-define
       mew-message-mode-menu
       mew-message-mode-map
       "Menu used in Mew message mode."
       mew-message-mode-menu-spec)
    )
  )

;;;
;;; Message mode
;;;

(defun mew-message-mode ()
  "Major mode for display a message.
The keys that are defined for this mode are:

SPC	Scroll up this message.
DEL	Back scroll this message.
n	Display below message or part.
p	Display above message or part.
h	Get back to summary mode.
a	Reply to this message.
A	Reply to this message and insert it in draft buffer.
f	Forward this message as MIME format.
r	Re-distribute this message with Resent-To: and Resent-Cc:.
d	Mark this message with delete mark(default is \"D\").
o	Mark this message or this part's message with refile
	mark(default is \"o\"). If already marked with \"o\", it
	prints to which folder this message is refiled.
!	
"
  (interactive)
  (setq major-mode 'mew-message-mode)
  (setq mode-name "Message")
  (use-local-map mew-message-mode-map)
  (setq buffer-read-only t)
  (setq mew-message-citation 'header)
  (make-local-variable 'page-delimiter)
  (setq page-delimiter mew-page-delimiter)
  (run-hooks 'mew-message-mode-hook)
  )

(defun mew-message-next-page (&optional lines)
;; return t   if no more pages
;;        nil if more pages
  (interactive)
  (move-to-window-line -1)
  (if (save-excursion
        (end-of-line)
        (and (pos-visible-in-window-p)
             (eobp)))
      ;; Nothing in this page.
      (if (or (null mew-break-pages)
	      (save-excursion
		(save-restriction
		  (widen) (forward-line) (eobp)))) ;Real end-of-buffer?
;;	  (progn (message "End of message") t) ;Nothing more.
	  t
	(mew-message-narrow-to-page 1)		;Go to next page.
	(run-hooks 'mew-message-hook)
	nil
	)
    ;; More in this page.
    (condition-case ()
	(scroll-up lines)
      (end-of-buffer
       ;; Long lines may cause an end-of-buffer error.
       (goto-char (point-max))))
    nil
    ))

(defun mew-message-narrow-to-page (&optional arg)
  (interactive "P")
  (setq arg (if arg (prefix-numeric-value arg) 0))
  (save-excursion
    (forward-page -1) ;;Beginning of current page.
    (forward-char 1)  ;; for compatibility with emacs-19.28 and emacs-19.29
    (widen)
    (cond
     ((> arg 0)	(forward-page arg))
     ((< arg 0) (forward-page (1- arg)))
     )
    (forward-page)
    (narrow-to-region (point)
		      (progn
			(forward-page -1)
			(if (and (eolp) (not (bobp)))
			    (forward-line))
			(point)))
    ))

(defun mew-message-prev-page (&optional lines)
  (interactive)
  (move-to-window-line 0)
  (if (and mew-break-pages
	   (bobp)
	   (not (save-restriction (widen) (bobp)))) ;Real beginning-of-buffer?
      (progn
	(mew-message-narrow-to-page -1) ;Go to previous page.
	(goto-char (point-max))
	(recenter -1))
    (scroll-down lines)))

(defun mew-message-goto-summary ()
  (interactive)
  (pop-to-buffer (car (mew-current-get 'message)))
  (if (cdr (mew-current-get 'message))
      (mew-summary-jump-message (cdr (mew-current-get 'message)))))

(defun mew-message-reply ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-reply))

(defun mew-message-reply-with-citation ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-reply-with-citation))

(defun mew-message-forward ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-forward))

(defun mew-message-redist ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-redist))
 
(defun mew-message-rmm ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-rmm))

(defun mew-message-refile ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-refile))

;;; hack hack 
;;; Now folder name and buffer name is different

(defun mew-message-next-msg (&optional arg)
  (interactive "p")
  (let* ((obuf (current-buffer))
	 (buf (window-buffer (previous-window)))
	 (buffer-read-only nil))
    (pop-to-buffer buf) ;; for the next forward-line
    (if (not (or (eq major-mode 'mew-summary-mode)
		 (eq major-mode 'mew-virtual-mode)))
	()
      (forward-line arg) ;; minus arg results in prev
      (mew-summary-display))
    (pop-to-buffer obuf) ;; for window config
    ))
      
(defun mew-message-prev-msg (&optional arg)
  (interactive "p")
  (mew-message-next-msg (- arg)))

(provide 'mew-message)
