;;;			     mew-mark.el
;;;
;;;		Copyright (C) 1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: March 23, 1997
;;; Revised: April 21, 1997
;;;

(defconst mew-mark-version "mew-mark.el version 0.02")

(require 'mew)

(defvar mew-mark-rmm    ?D)
(defvar mew-mark-mark   ?@)
(defvar mew-mark-hop    ?*)
(defvar mew-mark-refile ?o) ;; do you like "^"?
(defvar mew-mark-tmp    ?%) ;; temporary use only.

(defvar mew-mark-switch
  (list 
   (cons mew-mark-rmm    'mew-delete-line)
   (cons mew-mark-refile 'mew-delete-line)
   (cons mew-mark-mark   'mew-summary-mark-letitbe)
   (cons mew-mark-hop    'mew-summary-mark-letitbe)
   )
  )

(defvar mew-undo-switch
  (list 
   (cons mew-mark-rmm    'mew-summary-unmark)
   (cons mew-mark-refile 'mew-summary-refile-undo)
   (cons mew-mark-mark   'mew-summary-unmark)
   (cons mew-mark-hop    'mew-summary-unmark)
   )
  )

;;
;; mark
;;

(defun mew-summary-marked-p ()
  (save-excursion
      (beginning-of-line)
      (cond 
       ((looking-at (concat mew-summary-message-regex " \\|^$")) nil) ;; zzz
       (t t))))

(defun mew-summary-mark-exist-p (marklist)
  (let ((regex 
	 (concat
	  mew-summary-message-regex
	  "["
	  (mapconcat (function char-to-string) marklist "\\|")
	  "]"
	   )))
    (save-excursion
      (goto-char (point-min))
      (re-search-forward regex nil t)
      )
    ))

(defun mew-summary-mark (mark)
  (let ((buffer-read-only nil))
    (cond 
     ((null (mew-summary-marked-p))
      (beginning-of-line)
      (re-search-forward mew-summary-message-regex)
      (delete-char 1)
      (insert (char-to-string mark)))
     (t (message "Already marked")))
    ))

(defun mew-summary-unmark ()
  (save-excursion
    (let ((buffer-read-only nil))
      (beginning-of-line)
      (re-search-forward mew-summary-message-regex)
      (delete-char 1)
      (insert " ")
      )))

(defun mew-summary-mark-letitbe ()
  () ;; do nothing
  )

(defun mew-summary-mark-process (begin end)
  (let ((buffer-read-only nil)
	(m (make-marker)))
    (set-marker m end)
    (goto-char begin)
    (while (re-search-forward 
	    (concat mew-summary-message-regex "\\([^ +0-9]\\)") m t)
      (funcall (cdr (assoc (string-to-char (mew-match 2)) mew-mark-switch)))
      )
    (set-buffer-modified-p nil)
    ))

(defun mew-summary-mark-weak (mark)
  (let ((cmark (mew-summary-get-mark))
	(msg (mew-summary-message-number)))
    (cond 
     ((null msg) (message "No message"))
     ((or (equal cmark mew-mark-refile) (equal cmark mew-mark-rmm))
      (message "Already strongly marked as %s" (char-to-string cmark)))
     ((equal cmark mark)
      (message "Already marked as %s" (char-to-string cmark)))
     (t ;; no mark
      (save-excursion
	(if cmark (mew-summary-undo-one))
	(mew-summary-mark mark)))
     ))
  (set-buffer-modified-p nil)
  )

(defun mew-summary-mark-mark ()
  (interactive)
  (mew-summary-mark-weak mew-mark-mark)
  )

(defun mew-summary-mark-hop ()
  (interactive)
  (mew-summary-mark-weak mew-mark-hop)
  )

(defun mew-summary-mark-collect (mark &optional begin end)
  (save-excursion
    (let ((re (format (concat mew-summary-message-regex "%s") 
		      (regexp-quote (char-to-string mark))))
	  (msglist nil))
      (goto-char (if begin begin (point-min)))
      (while (re-search-forward re end t)
	(setq msglist (cons (mew-summary-message-number) msglist)))
      (nreverse msglist))))

(defun mew-summary-mark-collect2 (mark)
  (save-excursion
    (let ((re (format (concat mew-summary-message-regex "%s") 
		      (regexp-quote (char-to-string mark))))
	  (msglist nil))
      (goto-char (point-min))
      (while (re-search-forward re nil t)
	(setq msglist (cons 
		       (cons
			(mew-summary-folder-name)
			(mew-summary-message-number))
		       msglist)))
      (nreverse msglist) ;; return
      )))

(defun mew-summary-get-mark ()
  (let (mark)
    (save-excursion
      (beginning-of-line)
      (if (not (looking-at (concat mew-summary-message-regex "\\(.\\)")))
	  nil
	(setq mark (mew-match 2)) ;; regex includes \( \)
	(cond 
	 ((string-equal " " mark) nil)
	 (t (string-to-char mark))
	 )
	)
      )))

(defun mew-summary-mark-all ()
  (interactive)
  (let ((regex (concat mew-summary-message-regex " ")))
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward regex nil t)
	(mew-summary-mark mew-mark-hop))
      ))
  )

(defun mew-summary-mark-undo-all ()
  (interactive)
  (let ((regex (concat mew-summary-message-regex "[\\*oD]")))
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward regex nil t)
	(mew-summary-undo-one))
      ))
  )

(defun mew-summary-mark-regexp (regex)
  (interactive "sRegexp: ")
  (let ()
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward regex nil t)
	(if (not (mew-summary-marked-p))
	    (mew-summary-mark mew-mark-hop))
	)
      )
    )
  )

(defun mew-summary-refile-mark (folder &optional msg notmark)
  (let (msg tmp)
    (setq msg (or msg (mew-summary-message-number)))
    (setq mew-summary-buffer-refile
	  (if (setq tmp (assoc msg mew-summary-buffer-refile))
	      (if (mew-member folder (car (cdr tmp)))
		  mew-summary-buffer-refile
		  (cons 
		   (list msg (append (car (cdr tmp)) (list folder)))
		   (mew-alist-del mew-summary-buffer-refile msg)))
	    ; make new entry in mew-summary-buffer-refile
	    (cons (list msg (list folder))
		  mew-summary-buffer-refile)))
    (if notmark 
	()
      (mew-summary-mark mew-mark-refile))
    )
  )

(defun mew-decide-folder ()
  (interactive)
  (let (learn-info msg-buffer (tofolder nil) todir)
    (setq msg-buffer (or (mew-summary-display t)
			 (mew-buffer-message)))
    (save-excursion
      (set-buffer msg-buffer)
      (setq learn-info (mew-refile-folder-guess)))
    (setq tofolder (mew-input-folder (car learn-info)))
    (setq todir (mew-folder-to-dir tofolder))
    (if (mew-member todir mew-folder-list)
	tofolder
      (if (or (file-exists-p (concat mew-path "/" todir))
	      (mew-y-or-n-p 
	       (format "No folder %s exists. Create it? " tofolder)))
	  (progn
	    (setq 
	     mew-refile-alist (cons (mew-refile-pair todir) mew-refile-alist)
	     mew-folder-alist (cons (mew-folder-pair todir) mew-folder-alist)
	     mew-folder-list (cons todir  mew-folder-list))
	    (if (file-readable-p mew-folders-file)
		(save-excursion
		  (set-buffer (get-buffer-create mew-buffer-tmp))
		  (erase-buffer)
		  (insert todir)
		  (insert "\n")
		  (append-to-file (point-min) (point-max) mew-folders-file)))
	    tofolder ;; return value
	    )
	nil) ;; input is n
      ))
  )

(defun mew-summary-mark-refile ()
  (interactive)
  (let ((regex (concat mew-summary-message-regex
		       (regexp-quote (char-to-string mew-mark-hop))))
	(mew-summary-buffer-disp-msg nil)
	)
    (save-excursion
      (goto-char (point-min))
      (if (not (re-search-forward regex nil t))
	  (message "No marked messages")
	(beginning-of-line)
	(let ((mew-summary-refile-last-destination nil)
	      (mew-analysis nil)
	      folder)
	  (if (setq folder (mew-decide-folder))
	      (while (re-search-forward regex nil t)
		(let ((buffer-read-only nil));; mew-summary-unmark
		  (delete-backward-char 1)
		  (insert " "))
		(mew-summary-refile-mark folder)
		(forward-line))
	    ))
	)
      ))
  )

(defun mew-summary-exchange-mark (old-mark new-mark)
  (let ((regex (concat mew-summary-message-regex
		       (regexp-quote (char-to-string old-mark))))
	)
    (save-excursion
      (goto-char (point-min))
      (if (not (re-search-forward regex nil t))
	  (message "No marked messages")
	(beginning-of-line)
	(while (re-search-forward regex nil t)
	  (let ((buffer-read-only nil))
	    (delete-backward-char 1)
	    (insert (char-to-string new-mark)))
	  ))
      ))
  )
   
(defun mew-summary-mark-rmm ()		; * -> D
  (interactive)
  (mew-summary-exchange-mark mew-mark-hop mew-mark-rmm))

(defun mew-summary-mark-at ()		; * -> @
  (interactive)
  (mew-summary-exchange-mark mew-mark-hop mew-mark-mark))

(defun mew-summary-mark-asterisk ()	; @ -> *
  (interactive)
  (mew-summary-exchange-mark mew-mark-mark mew-mark-hop))

(defun mew-summary-swap-mark<->hop ()	; @ <-> *
  (interactive)
  (mew-summary-exchange-mark mew-mark-mark mew-mark-tmp)
  (mew-summary-exchange-mark mew-mark-hop mew-mark-mark)
  (mew-summary-exchange-mark mew-mark-tmp mew-mark-hop)
  )

;;
;; undo
;;

(defun mew-summary-undo (count)
  (interactive "P")
  (if (mew-summary-part-number)
      ;; go onto the current message anyway
      ;; mew-decode-syntax-delete saves excursion...
      (re-search-backward mew-summary-message-regex nil t nil))
  (mew-decode-syntax-delete)
  (save-excursion
    (if (not (numberp count))
	(mew-summary-undo-one)
      (while (> count 0)
	(mew-summary-undo-one)
	(forward-line)
	(setq count (1- count)))
      (while (< count 0)
	(mew-summary-undo-one)
	(forward-line -1)
      (setq count (1+ count)))
      )
    )
  (set-buffer-modified-p nil)
  )

(defun mew-summary-undo-one ()
  (interactive)
  (cond 
   ((mew-summary-marked-p)
    (beginning-of-line)
    (looking-at (concat mew-summary-message-regex "\\([^ +0-9]\\)"))
    (funcall (cdr (assoc (string-to-char (mew-match 2)) mew-undo-switch))))
   ((eobp) (message "No message"))
   (t (message "No mark"))
   )
  )

(defun mew-summary-undo-all ()
  (interactive)
  (let ((char nil) (ociea cursor-in-echo-area) (regex nil))
    (unwind-protect
	(progn
	  (message "Input mark : ")
	  (setq cursor-in-echo-area t)
	  (setq char (read-char))
	  (message "Input mark : %s" (char-to-string char))
	  )
      (setq cursor-in-echo-area ociea))
    (if (null (assoc char mew-mark-switch))
	(message "Mark %s is not supported" (char-to-string char))
      (mew-decode-syntax-delete)
      (message "Undoing %s ..." (char-to-string char))
      (save-excursion
	(goto-char (point-min))
	(setq regex (concat mew-summary-message-regex
			    (regexp-quote (char-to-string char))))
	(while (re-search-forward regex nil t)
	  (mew-summary-undo-one)) ;; xxxx
	)
      (message "Undoing %s ... done" (char-to-string char))
      )
    )
  (set-buffer-modified-p nil)
  )

(defun mew-summary-refile-undo ()
  (setq mew-summary-buffer-refile
	(mew-delq (assoc (mew-summary-message-number) 
			 mew-summary-buffer-refile)
		  mew-summary-buffer-refile))
  (mew-summary-unmark)
  )


;;
;; process control
;;

;; mew-summary-buffer-process is a key to see if exclusive

(defun mew-summary-exclusive-p ()
  (cond
   ((eq mew-summary-buffer-process t)
    (message "Try again later.")
    nil) ;; not exclusive
   ((processp mew-summary-buffer-process)
    (message "%sing now. Try again later."
	     (process-name mew-summary-buffer-process))
    nil) ;; not exclusive
   (t t)) ;; exclusive
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;

(defun mew-summary-rmm (count)
  (interactive "P")
  (if (mew-summary-part-number)
      ;; go onto the current message anyway
      ;; mew-decode-syntax-delete saves excursion...
      (re-search-backward mew-summary-message-regex nil t nil))
  (mew-decode-syntax-delete)
  (if (not (numberp count))
      (mew-summary-rmm-one)
    (while (> count 0)
      (mew-summary-mark mew-mark-rmm)
      (mew-decode-syntax-delete)
      (forward-line)
      (setq count (1- count)))
    (while (< count 0)
      (mew-summary-mark mew-mark-rmm)
      (mew-decode-syntax-delete)
      (forward-line -1)
      (setq count (1+ count)))
    ))

(defun mew-summary-rmm-one ()
  (let ((mark (mew-summary-get-mark))
	(msg (mew-summary-message-number)))
    (cond
     ((null msg) (message "No message"))
     ((equal mark mew-mark-rmm) (message "Already marked as rmm"))
     ((equal mark mew-mark-refile)
      (if (mew-y-or-n-p "Already marked as refile. Delete it? ")
	  (progn
	    (mew-summary-undo-one)
	    (save-excursion (mew-summary-mark mew-mark-rmm))
	    )))
     ((or (equal mark mew-mark-hop) (equal mark mew-mark-mark))
      (mew-summary-undo-one)
      (save-excursion (mew-summary-mark mew-mark-rmm)))
     (t ;; no mark
      (save-excursion (mew-summary-mark mew-mark-rmm))
      (mew-summary-display-next)
      )
     )))

(defun mew-summary-folder-mark-exec ()
  (if (mew-summary-mark-exist-p (list mew-mark-rmm mew-mark-refile))
      (if (mew-y-or-n-p "Marked messages exist. Process mark before scan? ")
	  (mew-summary-exec)
	)))

(defun mew-summary-exec-current ()
  (interactive)
  (let (beg end)
    (save-excursion
      (setq beg (progn (beginning-of-line) (point)))
      (setq end (progn (end-of-line) (point))))
    (mew-summary-exec beg end)))

(defun mew-summary-exec-region (r1 r2)
  (interactive "r")
  (mew-summary-exec r1 r2))

(defun mew-summary-exec (&optional r1 r2)
  (interactive)
  (message "Collecting marks ...")
  (let* ((begin (or r1 (point-min)))
	 (end (or r2 (point-max)))
	 (msgs (mew-summary-mark-collect mew-mark-refile begin end))
	 (dels (mew-summary-mark-collect mew-mark-rmm begin end))
	 (src (buffer-name))
	 refal folder folder-list tmp msg)
    (if (not (mew-summary-exclusive-p))
	()
      (if (not (or msgs dels))
	  (message "No marks")
	(mew-window-configure (current-buffer) 'summary)
	(setq mew-summary-buffer-process t) ;; this must be here
	(message "Refiling and deleting ...")
	(while msgs
	  (setq msg (car msgs))
	  (setq msgs (cdr msgs))
	  (if (setq folder-list 
		    (car (cdr (assoc msg mew-summary-buffer-refile))))
	      ;; first, remove current folder from folder-list
	      (if (mew-member src folder-list)
		  (save-excursion
		    (setq folder-list (mew-delq src folder-list))
		    (goto-char (point-min))
		    (re-search-forward (concat "^ *" msg) nil t)
		    (mew-summary-unmark)
		    ;; if folder-list still have folders, refile to the
		    ;; folders using "-link" option
		    (if (> (length folder-list) 0)
			(apply (function call-process)
			       mew-prog-refile nil nil nil
			       "-src" src
			       (append (list "-link") folder-list (list msg))
			       ))
		    )
		;; if multi-refile, I must refile one by one. -- nom
		(if (nth 1 folder-list)
		    (apply (function call-process)
			   mew-prog-refile nil nil nil
			   "-src" src
			   (append folder-list (list msg)))
		  ;; else (folder-list has only one folder)
		  (setq folder (car folder-list))
		  (if (setq tmp (assoc folder refal))
		      (setq refal (cons (append tmp (list msg))
					(mew-delq tmp refal)))
		    (setq refal (cons (list folder msg) refal))
		    )
		  )
		)
	    )
	  )
	(while refal
	  (apply (function call-process)
		 mew-prog-refile nil nil nil
		 "-src" src
		 (car (car refal))
		 (cdr (car refal)))
	  (setq refal (cdr refal)))
	(setq mew-summary-buffer-refile nil)
	(if dels
	    (if (or (equal src mew-trash-folder) (not mew-trash-folder))
		(apply (function call-process)
		       mew-prog-rmm nil nil nil
		       (buffer-name)
		       dels)
	      (while dels
		(apply (function call-process)
		       mew-prog-refile nil nil nil
		       "-src" src
		       (car dels)
		       mew-trash-folder nil)
		(setq dels (cdr dels)))))
	(mew-current-set 'message nil) ;;; hack
	(mew-refile-folder-guess-save)
	(let ((next t))
	  (if (mew-summary-marked-p)
	      (mew-summary-message-down)) ;; point moves
	  (if (mew-summary-message-number)
	      (setq next (mew-summary-message-number)))
	  (mew-decode-syntax-delete)
	  (mew-summary-mark-process begin end)
	  (mew-summary-jump-message next)
	  )
	(if mew-summary-cache-use (mew-summary-folder-cache-save))
	(setq mew-summary-buffer-process nil)
	(run-hooks 'mew-summary-exec-hook)
	(set-buffer-modified-p nil)
	(message "Refiling and deleting ... done")
	))
    ))


(provide 'mew-mark)
