;;;			    mew-header.el
;;;
;;;	      Copyright (C) 1996-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: October  2, 1996
;;; Revised: April   21, 1997
;;;

(defconst mew-header-version "mew-header.el version 0.08")

(require 'mew)

(defvar mew-header-max-depth 20
  "*A value to decide loop depth for header field syntax analysis.
It was known as mew-loop-depth.
See also mew-header-max-length.")


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Header functions
;;;

(defun mew-header-get-value (field)
  "currently, when no match, it returns nil."
  (let ((case-fold-search t)
	(start nil)
	(key nil)
	(match nil)
	(count 0)
	(ret nil)) ;; (concat nil "foo") -> "foo"
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (format "^-*$\\|^%s[ \t]*" field) nil t)
	  (if (>= count mew-header-max-depth)
	      (throw 'header ret)) ;; xxx warn to users?
	  (setq count (1+ count))
	  (setq key (mew-match 0))
	  (setq start (match-end 0))
	  (if (string-match "^-*$" key) (throw 'header ret))
	  (forward-line)
	  (while (looking-at mew-lwsp) (forward-line))
	  (setq match (mew-buffer-substring start (1- (point))))
	  (if (null ret)
	      (if (null (string= "" match))
		  (setq ret match))
	    (if (null (string= "" match))
		(setq ret (concat ret "," match))))
	)))
    ret
    ))
	  
(defun mew-header-get-line (field)
;; return  with ^J
;; currently, when no match, it returns nil.
  (let ((case-fold-search t)
	(start nil)
	(key nil)
	(count 0)
	(ret nil)) ;; (concat nil "foo") -> "foo"
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (format "^-*$\\|^%s" field) nil t)
	  (if (>= count mew-header-max-depth)
	      (throw 'header ret)) ;; xxx warn to users?
	  (setq count (1+ count))
	  (setq key (mew-match 0))
	  (setq start (match-beginning 0))
	  (if (string-match "^-*$" key) (throw 'header ret))
	  (forward-line)
	  (while (looking-at mew-lwsp) (forward-line))
	  (setq ret (concat ret (mew-buffer-substring start (point))))
	  ))
      )
    ret
    ))

(defun mew-header-delete-lines (fields)
  (let ((case-fold-search t)
	(regex (mew-make-field-regex fields))
	key start)
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (concat "^-*$\\|" regex) nil t)
	  (setq key (mew-match 0)
		start (match-beginning 0))
	  (if (string-match "^-*$" key) (throw 'header nil))
	  (forward-line)
	  (while (looking-at mew-lwsp) (forward-line))
	  (delete-region start (point))
	  ))
      )))

(defun mew-header-resent-lines (fields)
  (let ((case-fold-search t)
	(regex (mew-make-field-regex fields))
	key)
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (concat "^-*$\\|" regex) nil t)
	  (setq key (mew-match 0))
	  (if (string-match "^-*$" key) (throw 'header nil))
	  (beginning-of-line)
	  (insert "Prev-")
	  ))
      )))

(defun mew-header-insert-here (field &optional value noret)
  (if value
      (if noret
	  (insert (format "%s %s" field value))
	(insert (format "%s %s\n" field value)))
    (if noret
	(insert (format "%s" field)))
    (insert (format "%s\n" field)))
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Low-level functions to parse fields
;;;

;;
;; Get a canonical syntax
;;
;; '"Winnie (The) Pooh" (loves "honey") @ (somewhere in) England'
;; ->
;; '"Winnie (The) Pooh"@England'
;;
(defun mew-header-syntax (string)
  (let* ((i 0) (len (length string))
	 (par-cnt 0) (ret-cnt 0)
	 (ret (make-string len ?x))
	 quoted c)
    (while (< i len)
      (setq c (aref string i))
      (cond
       (quoted
	(if (char-equal c ?\") (setq quoted nil))
	(aset ret ret-cnt c)
	(setq ret-cnt (1+ ret-cnt))
	)
       ((> par-cnt 0)
	(cond
	 ((char-equal c ?\() (setq par-cnt (1+ par-cnt)))
	 ((char-equal c ?\)) (setq par-cnt (1- par-cnt)))
	 )
	)
       ((char-equal c ?\() (setq par-cnt 1))
       ((char-equal c ?\n))
       ((char-equal c ?\t))
       ((char-equal c ?\ ))
       (t 
	(if (char-equal c ?\") (setq quoted t))
	(aset ret ret-cnt c)
	(setq ret-cnt (1+ ret-cnt))
	)
       )
      (setq i (1+ i))
      )
    (substring ret 0 ret-cnt)
    ))

;;
;; Split a string by a separator.
;;
;; "Winnie-The-Pooh" -> "Winnie", "The", "Pooh"
;;
(defun mew-header-split (string sep)
 (let ((quote nil)
       (len (length string))
       (ret nil)
       (start 0)
       (n 0))
   (while (< n len)
     (cond
      ((char-equal (aref string n) ?\")
       (setq quote (not quote)))
      ((char-equal (aref string n) sep)
       (if (null quote)
	   (progn
	     (setq ret (cons (substring string start n) ret))
	     (setq start (1+ n))
	     )
	 ))
      )
     (setq n (1+ n))
     )
   (setq ret (cons (substring string start n) ret))
   (nreverse ret)
   ))

;;
;; Extract an address from <>.
;;
;; "Winnie The Pooh <wtp@uk>" -> "wtp@uk"
;;
(defmacro mew-header-extract-addr-list (list)
  (` (mapcar (function mew-header-extract-addr) (, list))))

(defun mew-header-extract-addr (str)
  "Extracts a real e-mail address from STR and returns it.
e.g. \"Mine Sakurai <m-sakura@ccs.mt.nec.co.jp>\"
  ->  \"m-sakura@ccs.mt.nec.co.jp\".
e.g. \"m-sakura@ccs.mt.nec.co.jp (Mine Sakurai)\"
  ->  \"m-sakura@ccs.mt.nec.co.jp\"."
  (cond ((string-match ".*<\\([^>]*\\)>" str) ;; .* to extract last <>
         (mew-match 1 str))
        ((string-match "\\([^ \t]*@[^ \t]*\\)" str)
         (mew-match 1 str))
        (t str)
    )
  )

;;
;; Delete a string after @.
;;
;; "Pooh@England" -> "Pooh"
;;
(defmacro mew-header-delete-at-list (list)
  (` (mapcar (function mew-header-delete-at) (, list))))

(defun mew-header-delete-at (string)
  (if (string-match "@.*:" string)
      (setq string (substring string (match-end 0) (length string)))
    (setq string (substring string 0 (string-match "%" string)))
    (setq string (substring string 0 (string-match "@" string)))
    (substring string 0 (string-match ":;" string))
    ))

;;
;; Delete null strings.
;;
;; "Winnie", "", "The", "", "Pooh"
;; ->
;; "Winnie", "The", "Pooh"
;;
(defun mew-header-delete-nullstring-list (list)
  (if (null list)
      ()
    (cond 
     ((string= (car list) "") 
      (mew-header-delete-nullstring-list (cdr list)))
     (t 
      (cons (car list) (mew-header-delete-nullstring-list (cdr list)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; High-level functions to parse fields
;;;

(defun mew-header-syntax-list (string)
  (mew-header-split
   (mew-header-syntax string)
   ?;))

(defun mew-header-user-collect (list &optional func)
  (let ((fields (mapcar (function mew-header-get-value) list))
	(users "") (count 0) (begin 0))
    (while fields
      (if (car fields) 
	  (setq users (concat users "," (car fields))))
      (setq fields (cdr fields)))
    (while (and (< count mew-header-max-depth) 
		(string-match "," users begin))
      (setq begin (match-end 0))
      (setq count (1+ count))
      )
    (if (equal count mew-header-max-depth)
	(setq users (substring users 0 begin)))
    (let ((list
	   (mew-header-delete-nullstring-list
	    (mew-header-delete-at-list
	     (mew-header-extract-addr-list
	      (mew-header-split
	       (mew-header-syntax users)
	       ?,))))))
      (if func
	  (mapcar func list)
	list))
    ))

(defun mew-header-address-collect (list)
  (let ((fields (mapcar (function mew-header-get-value) list))
	(users "") (count 0) (begin 0))
    (while fields
      (if (car fields) 
	  (setq users (concat users "," (car fields))))
      (setq fields (cdr fields)))
    (while (and (< count mew-header-max-depth) 
		(string-match "," users begin))
      (setq begin (match-end 0))
      (setq count (1+ count))
      )
    (if (>= count mew-header-max-depth)
	(progn
	  (message "Too many addresses, truncated after %d" 
		   mew-header-max-depth)
	  (ding)
	  (sit-for 3) ;; enough time to read?
	  ))
    (if (equal count mew-header-max-depth) 
	(setq users (substring users 0 begin)))
    (mew-header-delete-nullstring-list
     (mew-header-extract-addr-list
      (mew-header-split
       (mew-header-syntax users)
       ?,)))
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Alias functions
;;;

(defun mew-header-expand-alias-list (list)
  "The function of MH alias expansion. One \"ali\" command is executed
   for all the members of LIST. And, list is returned."
  (if (null list)
      ()
    (save-excursion
      ;; alias extraction
      (set-buffer mew-buffer-tmp)
      (erase-buffer)
      (apply (function call-process)
	     mew-prog-ali nil mew-buffer-tmp nil "-list" list)
      (goto-char (point-min))
      (setq list nil)
      (while (not (eobp))
	(let ((p (point))
	      (theline nil))
	  (end-of-line)
	  (setq theline (mew-buffer-substring p (point)))
	  (delete-region p (point))
	  (setq list (cons theline list))
	  (forward-line)))
      (nreverse list))
    ))

(defun mew-header-decrypters-collect (list)
  (let ((fields (mapcar (function mew-header-get-value) list))
	(users "") (count 0) (begin 0) a)
    (catch 'first
      (while fields
	(setq a (car fields) fields (cdr fields))
	(if (null a)
	    ()
	  (setq users a)
	  (throw 'first nil)
	  )))
    (while fields
      (if (car fields) 
	  (setq users (concat users "," (car fields))))
      (setq fields (cdr fields)))
    (while (and (< count mew-header-max-depth) 
		(string-match "," users begin))
      (setq begin (match-end 0))
      (setq count (1+ count))
      )
    (if (>= count mew-header-max-depth)
	(progn
	  (message "Too many addresses, truncated after %d" 
		   mew-header-max-depth)
	  (ding)
	  (sit-for 3) ;; enough time to read?
	  ))
    (if (equal count mew-header-max-depth) 
	(setq users (substring users 0 begin)))
    users
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; e-mail address canonical form 
;;;

(defmacro mew-header-canform-list (list)
  (` (mapcar (function mew-header-canform) (, list))))

(defun mew-header-canform (string)
  "Complete STRING with mew-mail-domain."
  (if (or (null mew-mail-domain) (string= mew-mail-domain ""))
      string
    (if (string-match "@" string)
	string
      (concat string "@" mew-mail-domain)))
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Header encoding and decoding
;;;

(defconst mew-base64-mask 63)
(defconst mew-base64-boundary1 26)
(defconst mew-base64-boundary2 52)
(defconst mew-base64-boundary3 62)
(defconst mew-base64-boundary4 63)

(defvar mew-header-encode-switch
  (list
   (cons "B" 'mew-header-encode-base64)
   (cons "Q" 'mew-header-encode-qp)
   )
  )

(defvar mew-header-decode-switch
  (list
   (cons "B" 'mew-header-decode-base64)
   (cons "Q" 'mew-header-decode-qp)
   )
  )

(defun mew-header-encode (str encode)
  (let* ((attr (mew-assoc encode mew-header-encode-switch 0 t))
	 (fun (cdr attr))
	 (charset (mew-charset-guess-string str))
	 (enstr))
    (if (string= charset "us-ascii")
	str
      (setq enstr (mew-cs-encode-string str charset 'charset))
      (concat "=?" charset "?" encode "?" (funcall fun enstr) "?="))
    ))

(defun mew-header-decode (str &optional del-ret)
  (if del-ret
      (while (string-match "\n[ \t]*" str)
	(setq str (concat (substring str 0 (match-beginning 0))
			  (substring str (match-end 0) (length str)))
	      )
	)
    )
  (while (string-match mew-header-decode-regex str)
    (let* ((charset (mew-match 1 str))
	   (encode (mew-match 2 str))
	   (enstr (mew-match 3 str))
	   (head (substring str 0 (match-beginning 0)))
	   (tail (substring str (match-end 0) (length str)))
	   (func (cdr (mew-assoc encode mew-header-decode-switch 0 t)))
	   (destr ""))
      (setq destr (mew-cs-decode-string (funcall func enstr) charset 'charset))
      (setq str (concat head destr tail))
      ))
  str
  )

;;
;; Base64 encoding
;;

(defun mew-header-encode-base64 (str256)
  (let* ((rest (% (length str256) 3))
	 (count 0)
	 zpad epad len end
	 (ret ""))
    (cond 
     ((= rest 0) (setq zpad 0 epad 0 end nil))
     ((= rest 1) (setq zpad 2 epad 2 end -2))
     ((= rest 2) (setq zpad 1 epad 1 end -1))
     )
    (setq str256 (concat str256 (make-string zpad 0)))
    (setq len (length str256))
    (while (< count len)
      (let ((char0 (aref str256 count))
	    (char1 (aref str256 (1+ count)))
	    (char2 (aref str256 (+ 2 count))))
	(setq ret
	      (concat
	       ret
	       (char-to-string (mew-base64-char64 (lsh char0 -2)))
	       (char-to-string
		(mew-base64-char64 (logior (lsh (logand char0 3) 4)
					   (lsh char1 -4))))
	       (char-to-string
		(mew-base64-char64 (logior (lsh (logand char1 15) 2)
					   (lsh char2 -6))))
	       (char-to-string (mew-base64-char64 (logand char2 63))) ;; xxx
	       )
	      )
	(setq count (+ count 3))
	))
    (concat (substring ret 0 end) (make-string epad ?=))
    ))
	      
(defun mew-base64-char256 (ch64)
  (cond
   ((null ch64) 0)
   ((and (<= ?A ch64) (<= ch64 ?Z)) (- ch64 ?A))
   ((and (<= ?a ch64) (<= ch64 ?z)) (+ (- ch64 ?a) mew-base64-boundary1))
   ((and (<= ?0 ch64) (<= ch64 ?9)) (+ (- ch64 ?0) mew-base64-boundary2))
   ((char-equal ch64 ?+) mew-base64-boundary3)
   ((char-equal ch64 ?/) mew-base64-boundary4)
   ((char-equal ch64 ?=) 0)
   )
  )

;;
;; Base64 decoding
;;

(defun mew-header-decode-base64 (str64)
  (let* ((len (length str64))
	 (pad 0)
	 (count 0)
	 (ret ""))
    (if (string-match "=+$" str64)
	(setq pad (- (match-end 0) (match-beginning 0))))
    (if (or (string-match "[^a-zA-Z0-9+/=]" str64)
	    (not (equal (* (/ len 4) 4) len))
	    (< pad 0)
	    (> pad 3))
	" **BASE64 ENCODING ERROR** " ;; return
      (while (< count len)
	(let ((char0 (mew-base64-char256 (aref str64 count)))
	      (char1 (mew-base64-char256 (aref str64 (1+ count))))
	      (char2 (mew-base64-char256 (aref str64 (+ 2 count))))
	      (char3 (mew-base64-char256 (aref str64 (+ 3 count)))))
	  (setq ret
		(concat
		 ret
		 (char-to-string
		  (logand (logior (lsh char0 2) (lsh char1 -4)) 255))
		 (char-to-string 
		  (logand (logior (lsh char1 4) (lsh char2 -2)) 255))
		 (char-to-string
		  (logand (logior (lsh char2 6) char3) 255))
		 )
		)
	  (setq count (+ count 4))
	  ))
    (if (equal pad 0)
	ret
      (substring ret 0 (- pad))
      ))))

(defun mew-base64-char64 (ch256)
  (aref "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/" ch256))

;;
;; Quoted-printable encoding
;;

(defun mew-char-to-qhex (char)
  (concat "="
	  (char-to-string (aref "0123456789ABCDEF" (lsh char -4)))
	  (char-to-string (aref "0123456789ABCDEF" (logand char 15)))))

(defun mew-header-encode-qp (str)
  (let ((len (length str))
	(count 0)
	(ch nil)
	(ret))
    (while (< count len)
      (setq ch (aref str count))
      (cond
       ((and (> ch 32)
	     (< ch 126)
	     (not (equal ch ?=))
	     (not (equal ch ??))
	     (not (equal ch ?_)) ;; space
	     )
	(setq ret (concat ret (char-to-string ch)))
	)
       ((equal ch 32)
	(setq ret (concat ret "_"))
	)
       (t 
	(setq ret (concat ret (mew-char-to-qhex ch)))
	)
       )
      (setq count (1+ count))
      )
    ret
    ))

;;
;; Quoted-printable decoding
;;

(defun mew-hexstring-to-int (hex)
  (let ((len (length hex))
	(count 0)
	(ch nil)
	(ret 0))
    (while (< count len)
      (setq ch (aref hex count))
      (cond
       ((and (<= ?0 ch) (<= ch ?9))
	(setq ret (+ (* ret 16) (- ch ?0))))
       ((and (<= ?A ch) (<= ch ?F))
	(setq ret (+ (* ret 16) (+ (- ch ?A) 10))))
       ((and (<= ?a ch) (<= ch ?f))
	(setq ret (+ (* ret 16) (+ (- ch ?a) 10))))
       )
      (setq count (1+ count))
      )
    ret
    ))

(defun mew-header-decode-qp (qpstr)
  (let ((start) (end))
    (while (string-match "_" qpstr)
      (aset qpstr (match-beginning 0) 32)) ;; 32 = space
    (while (string-match "=[0-9A-Z][0-9A-Z]" qpstr)
      (setq start (match-beginning 0))
      (setq end (match-end 0))
      (setq qpstr
	    (concat (substring qpstr 0 start)
		    (char-to-string (mew-hexstring-to-int 
				     (substring qpstr (1+ start) end)))
		    (substring qpstr end nil))))
    qpstr
    ))

(provide 'mew-header)
