;;;			     mew-func.el
;;;
;;;		Copyright (C) 1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: March 23, 1997
;;; Revised: April 21, 1997
;;;

(defconst mew-func-version "mew-func.el version 0.02")

(require 'mew)

(defvar mew-emacs-y-or-n-p-use nil
  "*SPC is y and RET is n if t. SPC is n and RET is y, otherwise.")

(defun mew-replace-character (string from to)
  (let ((len (length string))
	(cnt 0))
    (while (< cnt len)
      (if (char-equal (aref string cnt) from)
	  (aset string cnt to))
      (setq cnt (1+ cnt)))
    string
    ))

(defun mew-y-or-n-p (string)
  (interactive)
  (if mew-emacs-y-or-n-p-use
      (y-or-n-p string)
    (let ((char nil) (ociea cursor-in-echo-area))
      (unwind-protect
	  (progn
	    (setq cursor-in-echo-area t)
	    (catch 'break
	      (while t
		(message (concat string "(y=RET or n=SPC) "))
		(setq char (read-char))
		(cond
		 ((char-equal char ?y)
		  (setq char t)
		  (throw 'break nil))
		 ((char-equal char ?\r)
		  (setq char t)
		  (throw 'break nil))
		 ((char-equal char ?n)
		  (setq char nil)
		  (throw 'break nil))
		 ((char-equal char ?\ )
		  (setq char nil)
		  (throw 'break nil))
		 (t (ding))
		 )
		))
	    )
	(setq cursor-in-echo-area ociea)
	)
      char ;; return value
      ))
  )

(defun mew-member-match (str list)
  (let ((n 0))
    (catch 'member
      (while list
	(if (equal (downcase (car list)) (downcase str))
	    (throw 'member n))
	(setq list (cdr list))
	(setq n (1+ n))
	))
    ))

;; will be soon obsolete by 'delete'
(defun mew-delq (key list)
  (let* ((pointer (cons nil list))
	 (top pointer))
    (while (cdr pointer)
      (if (equal key (car (cdr pointer))) 
	  (progn
	    (setcdr pointer (cdr (cdr pointer)))
	    (setq pointer (cons nil nil)))
	(setq pointer (cdr pointer))))
    (cdr top)))

(defun mew-folder-to-dir (folder)
  (if (char-equal (aref folder 0) ?+)
      (substring folder 1 nil)
    folder)
  )

(defun mew-dir-to-folder (dir)
  (if (char-equal (aref dir 0) ?+)
      dir
    (concat "+" dir))
  )

(defmacro mew-draft-to-mime (draft)
  (` (concat mew-draft-mime-folder "/" (file-name-nondirectory (, draft)))))

(defun mew-make-folder (folder)
  (mew-make-directory (mew-folder-to-dir folder)))

(defun mew-make-directory (path)
  (let ((parent (directory-file-name (file-name-directory path))))
    (if (null (file-directory-p parent))
	(mew-make-directory parent))
    (if (and (file-exists-p path) (not (file-directory-p path)))
	(delete-file path))
    (make-directory path)
    ))

(defun mew-delete-directory-recursively (dir)
  (let ((files (directory-files dir t "^[^.]\\|^.[^.]")))
    (while files
      (cond
       ((file-directory-p (car files))
	(mew-delete-directory-recursively (car files)))
       (t
	(delete-file (car files)))
       )
      (setq files (cdr files))
      ))
  (delete-directory dir)
  )

;; I don't like such a procedural programming, but for max depth safety.
(defun mew-rassoq (a alist)
  (catch 'conscell
    (while alist
      (if (equal (cdr (car alist)) a)
	  (throw 'conscell (car alist)))
      (setq alist (cdr alist)))))

(defun mew-match (pos &optional string)
  (cond 
   ((stringp string)
    (substring string (match-beginning pos) (match-end pos)))
   (t 
    (mew-buffer-substring (match-beginning pos) (match-end pos)))
   ))

(defun mew-assoc (key alist nth match)
  (let ((case-fold-search t))
    (if match
	(mew-assoc-match key alist nth)
      (mew-assoc-equal key alist nth))))

(defun mew-assoc-match (key alist nth)
  (if (null alist) 
      ()
    (let* ((car (car alist))
	   (nthobj (nth nth car)))
      (if (or (and (stringp nthobj)
		   (let ((case-fold-search t))
		     (string-match key nthobj))) ;;; see mew-assoc-match2
	      (equal nthobj key)
	      (eq nthobj t))
	  car
	(mew-assoc-match key (cdr alist) nth)))))

(defun mew-assoc2 (key alist nth match)
  (let ((case-fold-search t))
    (if match
	(mew-assoc-match2 key alist nth)
      (mew-assoc-equal key alist nth))))

(defun mew-assoc-match2 (key alist nth)
  (if (null alist) 
      ()
    (let* ((car (car alist))
	   (nthobj (nth nth car)))
      (if (or (and (stringp nthobj)
		   (let ((case-fold-search t))
		     (string-match nthobj key))) ;;; see mew-assoc-match
	      (equal nthobj key)
	      (eq nthobj t))
	  car
	(mew-assoc-match2 key (cdr alist) nth)))))

(defun mew-assoc-equal (key alist nth)
  (if (null alist) 
      ()
    (let* ((car (car alist))
	   (nthobj (nth nth car)))
      (if (or (equal nthobj key)
	      (eq nthobj t))
	  car
	(mew-assoc-equal key (cdr alist) nth)))))

;;(defun mew-assoc2 (key alist nth exact)
;;  (let ((case-fold-search t))
;;    (cond
;;     ((null alist) ())
;;     ((and exact 
;;	   (stringp (nth nth (car alist)))
;;	   (equal (downcase key) (downcase (nth nth (car alist)))))
;;      (car alist))
;;     ((equal (nth nth (car alist)) key)
;;      (car alist))
;;     ((eq (nth nth (car alist)) t)
;;      (car alist))
;;     (t (mew-assoc key (cdr alist) nth exact))
;;     )
;;    ))

(defun mew-delete-line ()
  (beginning-of-line)
  (let ((start (point)))
    (forward-line)
    (delete-region start (point))
    ))

(defmacro mew-directory-empty-p (dir)
  (` (null (car (cdr (cdr (directory-files (, dir)))))))
  )

(defun mew-which (file path)
  (catch 'loop
    (while path
      (if (file-exists-p (expand-file-name file (car path)))
	  (throw 'loop (expand-file-name file (car path)))
	(setq path (cdr path)))
      )
    ))

(defun mew-expand-file-name (msg &optional buffer-or-name)
  (if (bufferp buffer-or-name)
      (setq buffer-or-name (buffer-name buffer-or-name)))
  (and buffer-or-name 
       (setq buffer-or-name (directory-file-name buffer-or-name)))
  (if buffer-or-name
      (expand-file-name 
       (concat (mew-folder-to-dir buffer-or-name) "/" (mew-folder-to-dir msg))
       mew-path)
    (expand-file-name (mew-folder-to-dir msg) mew-path))
  )


(defun mew-subsequence (seq beg &optional end)
  (cond
   ((vectorp seq)
    (mew-subvector seq beg end))
   ((stringp seq)
    (substring  seq beg end))
   ((listp seq)
    (mew-sublist seq beg end))
   (t nil)
   )
  )

(defun mew-sublist (list beg &optional end)
  (let (i ret)
    (if (null end) (setq end (length list)))
    (setq end (1- end))
    (setq i end)
    (while (<= beg i)
      (setq ret (cons (nth i list) ret))
      (setq i (1- i)))
    ret
    )
  )

(defun mew-subvector (vec beg &optional end)
  (let (i j ret)
    (if (null end) (setq end (length vec)))
    (setq ret (make-vector (- end beg) nil))
    (setq i beg j 0)
    (while (< i end)
      (aset ret j (aref vec i))
      (setq i (1+ i) j (1+ j)))
    ret
    )
  )


(provide 'mew-func)
