;;;			      mew-ext.el
;;;
;;;	      Copyright (C) 1996-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: November 13, 1996
;;; Revised: April    21, 1997
;;;

(defconst mew-ext-version "mew-ext.el version 0.12")

(require 'mew)
(eval-when-compile
  (cond
   ((mew-which "efs.el" load-path)
    (require 'efs))
   ((mew-which "ange-ftp.el" load-path)
    (require 'ange-ftp))
   )
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; customize variables
;;;

(defvar mew-ext-prog-url "netscape")
(defvar mew-ext-anon-ftp-method 'ftp
  "A method to get the message body for access-type=anon-ftp.
If 'ftp is specified, ange-ftp or efs is used. If 'http is specified,
w3 is used.")

;;
;; encode
;;

(defvar mew-ext-default-access-type "anon-ftp")

(defvar mew-ext-ftp-server-list 
  '("ftp.aist-nara.ac.jp" "ftp.csce.kyushu-u.ac.jp" "sh.wide.ad.jp")
  )

(defvar mew-ext-encode-switch
  '(("ftp"         . mew-ext-encode-ftp)
;;    ("tftp"        . mew-ext-encode-tftp)
    ("anon-ftp"    . mew-ext-encode-anon-ftp)
    ("local-file"  . mew-ext-encode-local-file)
    ("mail-server" . mew-ext-encode-mail-server)
    ("url"         . mew-ext-encode-url))
  )

;;
;; decode
;;

(defvar mew-ext-switch
  '(("ftp"         . mew-ext-ftp)
    ("tftp"        . mew-ext-tftp)
    ("anon-ftp"    . mew-ext-anon-ftp)
    ("mail-server" . mew-ext-mail-server)
    ("url"         . mew-ext-url)) ;; RFC2017
  )

(defvar mew-ext-include-switch 
  '(("local-file" . mew-ext-include-local-file))
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; environment variables
;;;

(defvar mew-buffer-ext " *mew ext*")
(defvar mew-ext-suffix ".ext")

(cond
 ((mew-which "efs.el" load-path)
  (defun mew-ext-file-name-completion (file path)
    (require 'efs)
    (let ((efs-tmp-name-template mew-temp-file))
      (efs-file-name-completion file path)
      ))
  (defun mew-ext-file-name-all-completions (file path)
    (require 'efs)
    (let ((efs-tmp-name-template mew-temp-file))
      (efs-file-name-all-completions file path)
      ))
  (defun mew-ext-expand-dir (host user dir)
    (require 'efs)
    (let ((efs-tmp-name-template mew-temp-file) exp)
      (setq exp (efs-expand-file-name (format "/%s@%s:%s" user host dir)))
      (if (string-match ".*:\\(.*\\)$" exp)
	  (mew-match 1 exp))
      ))
  (defun mew-ext-copy-file-internal (remote local passwd)
    (require 'efs)
    (let ((efs-tmp-name-template mew-temp-file)
	  (efs-generate-anonymous-password passwd)
	  (parsed (efs-ftp-path remote)))
      (efs-copy-file-internal remote parsed local nil 
			      nil nil nil nil t 'image)
      ))
  )
 ((mew-which "ange-ftp.el" load-path)
  (defun mew-ext-file-name-completion (file path)
    (require 'ange-ftp)
    (let ((ange-ftp-tmp-name-template mew-temp-file))
      (ange-ftp-file-name-completion file path)
      ))
  (defun mew-ext-file-name-all-completions (file path)
    (require 'ange-ftp)
    (let ((ange-ftp-tmp-name-template mew-temp-file))
      (ange-ftp-file-name-all-completions file path)
      ))
  (defun mew-ext-expand-dir (host user dir)
    (require 'ange-ftp)
    (let ((ange-ftp-tmp-name-template mew-temp-file) exp)
      (setq exp (ange-ftp-expand-file-name (format "/%s@%s:%s" user host dir)))
      (if (string-match ".*:\\(.*\\)$" exp)
	  (mew-match 1 exp))
      ))
  (defun mew-ext-copy-file-internal (remote local passwd)
    (require 'ange-ftp)
    (let ((ange-ftp-tmp-name-template mew-temp-file)
	  (ange-ftp-generate-anonymous-password passwd))
      (ange-ftp-copy-file-internal remote local t nil nil nil t)
      ))
  )
 )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Encode
;;;

(defun mew-attach-external-body ()
  (interactive)
  (if (not (mew-attach-not-line012-1))
      (message "Can't insert external-body here")
    (let* ((nums (mew-attach-nums))
	   (subdir (mew-attach-expand-path mew-encode-syntax nums))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (file (concat (mew-attach-random-filename) mew-ext-suffix))
	   (cnt 0) (max (length mew-attach-random-filename))
	   filepath ct)
      ;; mimedir / {subdir/} dir
      (if (not (equal subdir "")) 
	  (setq mimedir (expand-file-name subdir mimedir)))
      ;; mimedir / file
      (setq filepath (expand-file-name file mimedir))
      (while (and (file-exists-p filepath) (< cnt max))
	(setq file (concat (mew-attach-random-filename) mew-ext-suffix))
	(setq filepath (expand-file-name file mimedir))
	(setq cnt (1+ cnt)))
      (if (file-exists-p filepath)
	  (message "Could not make a file for external-body, sorry.")
	(setq ct (mew-ext-encode filepath))
	(setq mew-encode-syntax
	      (mew-syntax-insert-entry
	       mew-encode-syntax 
	       nums
	       (mew-encode-syntax-single file ct)))
	(mew-encode-syntax-print mew-encode-syntax)
	))
    ))

(defun mew-ext-encode (filename)
  (let (buf ret access-type ct name)
    (save-excursion
      (set-buffer (get-buffer-create mew-buffer-ext))
      (setq buf (current-buffer))
      (erase-buffer)
      ;;content-header
      (setq access-type (mew-input "Access type" mew-ext-encode-switch t 
				   mew-ext-default-access-type))
      (setq ret (funcall (cdr (assoc access-type mew-ext-encode-switch))))
      ;;message-header
      (cond 
       ((string-equal access-type "url")
	(setq name "URL")
	(setq ct "Text/Html"))
       ((string-equal access-type "mail-server")
	(setq name "Mail server's file")
	(setq ct mew-ct-apo))
       (t 
	(setq name (file-name-nondirectory
		    (mew-header-syntax (mew-syntax-get-member ret "name"))))
	;; name is quoted
	(setq ct (capitalize (mew-file-content
			      (mew-file-attr name mew-mime-content-type)))))
       )
      (setq ct (mew-input-type "Type for %s (%s): " name ct
			       mew-mime-content-type-list))
      (mew-header-insert-here mew-ct: ct)
      (mew-header-insert-here "Message-ID:" (mew-create-message-id))
      (insert "\n")
      (if (not (string= access-type "mail-server"))
	  () ;; message-body is not necessary
	;;message-body
	(insert (read-string "Input message to the mail-server: "))
	(insert "\n")
	)
      (write-file filename)
      )
    (kill-buffer buf)
    (cons mew-ct-ext (cons (format "access-type=\"%s\"" access-type) ret))
    ))

(defun mew-ext-encode-ftp ()
  ;; "name" "site" "directory" "mode"
  (let ((mew-ext-host (mew-input "FTP server"
				 (if mew-ext-ftp-server-list 
				     (mapcar (function list) 
					     mew-ext-ftp-server-list))
				 nil
				 (car mew-ext-ftp-server-list)))
	mew-ext-user path dir file ret)
    (setq ret (list (format "site=\"%s\"" mew-ext-host)))
    (setq mew-ext-user (read-string (format "User name at %s: " mew-ext-host)
				(user-login-name)))
    (setq path (mew-input-rfile "Filename :"))
    (setq file (file-name-nondirectory path))
    (setq dir (file-name-directory path))
    (if (string-match "~" dir)
	(setq dir (mew-ext-expand-dir mew-ext-host mew-ext-user dir)))
    (cond
     (dir
      (setq ret (cons (format "directory=\"%s\"" dir) ret))
      (setq ret (cons (format "name=\"%s\"" file) ret)))
     (t
      (setq ret (cons (format "name=\"%s\"" file) ret))
      )
     )
    ret
    ))

(defun mew-ext-encode-tftp () ;; xxx
  ;; not yet
  )

(defun mew-ext-encode-anon-ftp ()
  ;; "name" "site" "directory" "mode"
  (let ((mew-ext-user "anonymous")
	(mew-ext-host (mew-input "FTP server"
				 (if mew-ext-ftp-server-list 
				     (mapcar (function list) 
					     mew-ext-ftp-server-list))
				 nil
				 (car mew-ext-ftp-server-list)))
	path dir file ret)
    (setq ret (list (format "site=\"%s\"" mew-ext-host)))
    (setq path (mew-input-rfile "Filename :"))
    (setq file (file-name-nondirectory path))
    (setq dir (file-name-directory path))
    (if (string-match "~" dir)
	(setq dir (mew-ext-expand-dir mew-ext-host mew-ext-user dir)))
    (cond
     (dir
      (setq ret (cons (format "directory=\"%s\"" dir) ret))
      (setq ret (cons (format "name=\"%s\"" file) ret)))
     (t
      (setq ret (cons (format "name=\"%s\"" file) ret))
      )
     )
    ret
    ))

(defun mew-ext-encode-local-file ()
  ;; "name" "site"
  (let ((file (read-file-name "File name: ")))
    (list (format "name=\"%s\"" (expand-file-name file)))
    ))

(defun mew-ext-encode-mail-server ()
  ;; "server" "subject"
  (let (server subject)
    (setq server (mew-input-address "Server address:"))
    (setq subject (read-string "Subject: "))
    (list (format "server=\"%s\"" server)
	  (format "subject=\"%s\"" subject))
    ))
   
(defun mew-ext-encode-url ()
  ;; "url"
  (let ((url (read-string "URL: ")))
    (list (format "url=\"%s\"" url))
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Decode
;;;

;;
;; exclude
;;

(defun mew-mime-external-body (begin end params &optional execute)
  ;; message-buffer
  (let* ((access-type (mew-syntax-get-member params "access-type"))
	 (program (cdr (mew-assoc access-type mew-ext-switch 0 t))))
    ;; xxx expire
    (if (and (symbolp program) (fboundp program))
	(funcall program begin end params execute))
    ))

(defun mew-ext-ftp (begin end params execute)
  (let* ((site (mew-syntax-get-member params "site"))
	 (directory (mew-syntax-get-member params "directory"))
	 (name (mew-syntax-get-member params "name"))
	 (size (mew-syntax-get-member params "size"))
	 (filepath nil) (localfile nil) (remotefile nil) (getit t)
	 (username "") (buffer-read-only nil))
    (if directory
	(if (string-match ".*/$" directory) ;; xxx Win95???
	    (setq filepath (concat directory name))
	  (setq filepath (concat directory "/" name)))
      (setq filepath name))
    (erase-buffer)
    (insert " ####### ####### ######  \n"
	    " #          #    #     # \n"
	    " #          #    #     # \n"
	    " #####      #    ######  \n"
	    " #          #    #       \n"
	    " #          #    #       \n"
	    " #          #    #       \n"
            "\n\n")
    (insert "You can get the message content by FTP\n\n"
            (format "SITE:\t%s\n" site)
            (format "FILE:\t%s\n" filepath))
    (if size (insert (format "SIZE:\t%s bytes\n" size)))
    (if (not (and execute (mew-y-or-n-p "Get the message? ")))
	(insert "\nTo get this file, type "
		(substitute-command-keys
		 "\\<mew-summary-mode-map>\\[mew-summary-execute-external]."))
      (setq username (read-string (format "User name at %s: " site)
				  (user-login-name)))
      (setq remotefile (format "/%s@%s:%s" username site filepath))
      (setq localfile
	    (expand-file-name 
	     (read-file-name "Save to: " 
			     (format "%s%s" default-directory name))))
      (if (file-exists-p localfile)
	  (if (mew-y-or-n-p (format "The local file exists. Overwrite? "))
	      (delete-file localfile)
	    (setq getit nil)
	    (message "The file wasn't retrieved")))
      (if getit (mew-ext-copy-file-internal remotefile localfile nil))
      )
    ))

(defun mew-ext-tftp (begin end params execute)
  (message "access-type TFTP is not supported yet")
  )

(defun mew-ext-anon-ftp (begin end params execute)
  (let* ((site (mew-syntax-get-member params "site"))
	 (directory (mew-syntax-get-member params "directory"))
	 (name (mew-syntax-get-member params "name"))
	 (size (mew-syntax-get-member params "size"))
	 (getit t) (buffer-read-only nil)
	 filepath localfile remotefile url)
    (if directory
	(if (string-match ".*/$" directory) ;; xxx Win95???
	    (setq filepath (concat directory name))
	  (setq filepath (concat directory "/" name)))
      (setq filepath name))
    (erase-buffer)
    (insert " Anonymous \n"
            " ####### ####### ######  \n"
	    " #          #    #     # \n"
	    " #          #    #     # \n"
	    " #####      #    ######  \n"
	    " #          #    #       \n"
	    " #          #    #       \n"
	    " #          #    #       \n"
            "\n\n")
    (insert "You can get the message content by FTP\n\n"
            (format "SITE:\t%s\n" site)
            (format "FILE:\t%s\n" filepath))
    (if size (insert (format "SIZE:\t%s bytes\n" size)))
    (if (null (and execute (mew-y-or-n-p "Get the message? ")))
	(insert "\nTo get this file, type "
		(substitute-command-keys
		 "\\<mew-summary-mode-map>\\[mew-summary-execute-external]."))
      (setq remotefile (format "/%s@%s:%s" "anonymous" site filepath))
      (setq url (format "ftp://%s%s" site filepath))
      (setq localfile
	    (expand-file-name 
	     (read-file-name "Save to: "
			     (format "%s%s" default-directory name))))
      (if (file-exists-p localfile)
	  (if (mew-y-or-n-p (format "The local file exists. Overwrite? "))
	      (delete-file localfile)
	    (setq getit nil)
	    (message "The file wasn't retrieved"))
	)
      (if (not getit)
	  ()
	(cond
	 ((eq mew-ext-anon-ftp-method 'ftp)
	  (mew-ext-copy-file-internal remotefile localfile mew-mail-address))
	 ((eq mew-ext-anon-ftp-method 'http)
	  (require 'w3)
	  (w3-fetch url))
	 )
	)
      )
    ))

(defun mew-ext-mail-server (begin end params execute)
  (let ((server (mew-syntax-get-member params "server"))
	(subject (mew-syntax-get-member params "subject"))
	(size (mew-syntax-get-member params "size"))
	(start nil)
	(buffer-read-only nil))
    (erase-buffer)
    (insert "#     #    #      ###   #\n"
            "##   ##   # #      #    #\n"
            "# # # #  #   #     #    #\n"
            "#  #  # #     #    #    #\n"
            "#     # #######    #    #\n"
            "#     # #     #    #    #\n"
            "#     # #     #   ###   #######\n"
            "\n\n")
    (insert "You can get the message by e-mail\n"
            "\n"
            (format "SERVER:\t\t%s\n" server))
    (if size (insert (format "SIZE:\t%s bytes\n" size)))
    (save-excursion
      (set-buffer (mew-current-get 'cache))
      (save-restriction
	(narrow-to-region begin end)
	(goto-char (point-min))
	;; find a phantom body (in RFC1521)
	(re-search-forward "^$" nil t)
	(forward-line)
	(setq start (point))
	))
    ;; pickd up source from 'mew-send
    (if (null (and execute
		   (mew-y-or-n-p (format "Send a message to %s? " server))))
	(insert "\nTo send this mail, type "
		(substitute-command-keys
		 "\\<mew-summary-mode-map>\\[mew-summary-execute-external]."))
      (mew-summary-send nil server nil subject)
      (insert-buffer-substring (mew-current-get 'cache) start end)
      (mew-draft-make-mime)
      (if (mew-y-or-n-p "Send this message? ")
	  (mew-draft-real-send-letter)
	))
    ))

(defun mew-ext-url (begin end params execute)
  (let ((url (mew-syntax-get-member params "url"))
	(size (mew-syntax-get-member params "size"))
	(process-connection-type mew-connection-type1)
	(buffer-read-only nil))
    (erase-buffer)
    (insert "#     # ######  #\n"
            "#     # #     # #\n"
            "#     # #     # #\n"
            "#     # ######  #\n"
            "#     # #   #   #\n"
            "#     # #    #  #\n"
            " #####  #     # #######\n"
            "\n\n")
    (insert (format "You can gain access to the url with \"%s\"\n" 
		    mew-ext-prog-url)
	    "\n"
	    (format "URL:\t\t%s\n" url))
    (if size (insert (format "SIZE:\t%s bytes\n" size)))
    (if (and execute
	     (mew-y-or-n-p (format "Gain access to %s ?" url)))
	(start-process
	 (format "*mew %s*" mew-ext-prog-url)
	 mew-buffer-tmp mew-ext-prog-url url)
      (insert "\nTo show this URL, type "
	      (substitute-command-keys
	       "\\<mew-summary-mode-map>\\[mew-summary-execute-external]."))
      )
    ))

;;
;; include
;;

(defun mew-ext-include-local-file (params)
  (mew-flet
   (let* ((file (mew-syntax-get-member params "name"))
	  (size (mew-file-size file)))
     (if (file-exists-p file)
	 (if (> size mew-file-max-size)
	     (if (mew-y-or-n-p
		  (format "This mail size is %s. Truncate it? " size))
		 (insert-file-contents file nil 0 mew-file-max-size)
	       (insert-file-contents file))
	   (insert-file-contents file)))
     ))
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Misc
;;;

(defun mew-input (prompt alist &optional require-match initial)
  (let* ((completion-ignore-case t)
	 (question (if initial (format "%s (%s) : " prompt initial)
		     (format "(%s) : " prompt)))
	 (value (completing-read question alist nil require-match nil)))
    (if (and initial (string= value "")) initial value)
    ))

(defun mew-create-message-id ()
  ;; this is not unique if used with very short interval.
  ;; but it's ok
  (format "%s.%s.%s@%s" (nth 0 (current-time)) (nth 1 (current-time)) 
	  (emacs-pid) (system-name))
  )

(defvar mew-ext-host "")
(defvar mew-ext-user "")
(defvar mew-ext-map nil)

(if mew-ext-map
    ()
  (setq mew-ext-map (make-sparse-keymap))
  (define-key mew-ext-map "\t"     'mew-ext-rfile)
  (define-key mew-ext-map "\r"     'exit-minibuffer)
  (define-key mew-ext-map "\n"     'exit-minibuffer)
  (define-key mew-ext-map "\C-g"   'abort-recursive-edit)
  )

(defun mew-input-rfile (prompt) ;; prompt="To:"
  (read-from-minibuffer (concat prompt " ") "" mew-ext-map nil)
  )

(defun mew-ext-rfile ()
  (interactive)
  (let (file path comp all)
    (if (re-search-backward "/" nil t)
	(forward-char 1)
      (beginning-of-line))
    (setq file (buffer-substring (point) (point-max)))
    (delete-region (point) (point-max))
    (setq path (format "/%s@%s:%s"
		       mew-ext-user
		       mew-ext-host
		       (buffer-substring (point-min) (point))))
    (setq comp (mew-ext-file-name-completion file path))
    (setq all (mew-ext-file-name-all-completions file path))
    (cond
     ((eq comp t) 
      (insert file)
      (and (get-buffer mew-buffer-completions)
	   (kill-buffer mew-buffer-completions))
      )
     ((equal 1 (length all))
      (insert (car all))
      (and (get-buffer mew-buffer-completions)
	   (kill-buffer mew-buffer-completions))
      )
     ((stringp comp)
      (insert comp)
      (and (> (length all) 1)
	   (with-output-to-temp-buffer mew-buffer-completions
	     (display-completion-list all))))
     (t (insert file))
     )
    )
  )

(provide 'mew-ext)
