;;;			    mew-encode.el
;;;
;;;	      Copyright (C) 1996-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: October  2, 1996
;;; Revised: April   21, 1997
;;;

(defconst mew-encode-version "mew-encode.el version 0.13")

(require 'mew)

(defvar mew-print-function (function lpr-buffer))

(defvar mew-draft-backup-file ".mew-backup")
(defvar mew-draft-syntax-file ".mew-syntax")
(defvar mew-draft-coverpage "CoverPage")

(defconst mew-b64 "base64")
(defconst mew-qp  "quoted-printable")
(defconst mew-xg  "x-gzip64")

(defvar mew-prog-mime-encode-switch
  (list
   (cons mew-b64 '("-b"))
   (cons mew-qp  '("-q"))
   (cons mew-xg  '("-g"))
   )
  )

(defvar mew-prog-mime-encode-text-switch
  (list
   (cons mew-b64 '("-b" "-t"))
   (cons mew-qp  '("-q"))
   (cons mew-xg  '("-g" "-t"))
   )
  )


(defvar mew-encode-multipart-encrypted-switch
  '(("application/pgp-encrypted"  . mew-pgp-encrypt)
    ("application/moss-keys"      . mew-moss-encrypt))
  )

(defvar mew-encode-multipart-signed-switch
  '(("application/pgp-signature"  . mew-pgp-sign)
    ("application/moss-signature" . mew-moss-sign))
  )

;;;
;;;
;;;

(defun mew-draft-make-mime ()
  (interactive)
  (if (mew-header-get-value mew-ct:)
      (progn
	(ding)
	(message "%s already exists!" mew-ct:)
	)
    (if (mew-attach-p)
	(mew-draft-make-multi)
      (mew-draft-make-single))
    (goto-char (point-min))
    (sit-for 0)
    (re-search-forward mew-eoh2)
    (beginning-of-line)
    (setq mew-draft-buffer-header (point-marker)) ;; just in case
    ))

(defun mew-draft-make-single ()
  (mew-draft-refresh)
  (goto-char (marker-position mew-draft-buffer-header))
  ;; the beginning of "----"
  (let ((beg (point)))
    (forward-line)
    ;; cursor is just after "----\n"
    (delete-region beg (point))
    )
  (mew-encode-singlepart (mew-encode-syntax-single "text-file") nil nil t)
  )

(defun mew-encode-escape-region (beg end)
  (save-excursion
    (save-restriction
      (narrow-to-region beg end)
      (let ((case-fold-search nil))
	(re-search-forward "^From \\|^\\.\n" nil t)
	))
    ))
 
(defun mew-encode-get-topts (cte)
  (cdr (mew-assoc cte mew-prog-mime-encode-text-switch 0 t)))

(defun mew-encode-text-magic (file ctl cte)
  ;; return (file charset cte encopts)
  (let* ((charset (mew-syntax-get-member ctl "charset"))
	 ;; charset exists only when buffered is nil
	 encopts)
    (if (eq file t) (setq file nil)) ;; file is t if buffered
    (if (and charset cte)
	(list file charset cte (mew-encode-get-topts cte))
      (if file
	  ;; Even if charset is specified, we need to check the entire
	  ;; text object for cte to escape "\nFrom ".
	  (mew-frwlet
	   (or (mew-charset-to-symbol charset) mew-cs-infile) mew-cs-noconv
	   ;; Input the text file converting its charset to the internel.
	   ;; If charset is specified, use it.
	   ;; If mew-cs-infile is *autoconv*/'autodetect, charset is guessed
	   ;; but its rule depends on the user environment.
	   ;; For example, for usual Japanese, file is assumed as
	   ;; ISO-2022-JP, EUC-Japan, or SJIS on Mule.
	   (insert-file-contents file)
	   )
	() ;; buffered
	)
      ;; Let's guess charset if not specified. The charset of CoverPage 
      ;; for just singlepart(buffered) is always guessed.
      ;; Even if the file is EUC-Japan, charset is guessed as ISO-2022-JP.
      ;; Even if Latin1 is encoded in ISO-2022-JP-2, 
      ;; charset is guessed as ISO-8859-1.
      (if (null charset) 
	  (setq charset (mew-charset-guess-region (point) (point-max))))
      (if cte
	  (setq encopts (mew-encode-get-topts cte))
	(if (mew-charset-8bit-p charset)
	    (setq cte mew-qp encopts (mew-encode-get-topts mew-qp))
	  (if (mew-charset-iso-2022-p charset)
	      ;; Escaping "^From " with quoted-printable is harmful for
	      ;; ISO 2022 character set.
	      (setq cte "7bit" encopts nil)
	    (if (mew-encode-escape-region (point) (point-max))
		;; Probably us-ascii. Escape "^From " with quoted-printable.
		(setq cte mew-qp encopts (mew-encode-get-topts mew-qp))
	      (setq cte "7bit" encopts nil)))))
      (if (null encopts)
	  (setq file nil) ;; Not encode a file
	(if (not file)
	    (mew-frwlet
	     mew-cs-noconv (mew-charset-to-symbol charset)
	     (if (not (file-exists-p mew-temp-dir))
		 (mew-make-directory mew-temp-dir)) ;; just in case
	     (setq file (make-temp-name mew-temp-file))
	     ;; Write the text as MIME charset
	     ;; from internal to MIME charset (e.g. ISO-2022-JP)
	     ;; NEVER use call-process-region for privary reasons
	     (write-region (point) (point-max) file)))
	;; the file is used, so delete the region.
	(delete-region (point) (point-max)))
      (list file charset cte encopts)
      )))

(defun mew-encode-singlepart (syntax &optional path depth buffered)
  ;; path is nil if called make-single or security multipart
  ;; buffered is t if called make-single
  (let* ((file (mew-expand-file-name (mew-syntax-get-file syntax) path))
	 (ctl (mew-syntax-get-ct syntax))
	 (ct (capitalize (car ctl))) ;; just in case
	 (params (cdr ctl))
	 (cte (mew-syntax-get-cte syntax))
	 (cd (mew-syntax-get-cd syntax))
	 (privacy (mew-syntax-get-privacy syntax))
	 (beg (point))
	 (text-sw mew-prog-mime-encode-text-switch)
	 (binary-sw mew-prog-mime-encode-switch)
	 charset fcce encopts)
    (if (null (string-match "Text/" ct))
	;; No need to guess charset.
	;; The text based object is encoded by mewencode replacing LF 
	;; with CRLF. This is NOT identical to line break replacement for 
	;; object to be signed below.
	(if (null cte)
	    (setq cte "7bit")
	  (if (mew-member-match ct mew-mime-content-type-text-list)
	      (setq encopts (cdr (mew-assoc-match cte text-sw 0)))
	    (setq encopts (cdr (mew-assoc-match cte binary-sw 0)))))
      ;; Need to guess charset
      (setq fcce (mew-encode-text-magic (or buffered file) ctl cte))
      (setq file (nth 0 fcce) charset (nth 1 fcce) 
	    cte (nth 2 fcce) encopts (nth 3 fcce))
      ;; file is nil if buffered
      (setq params (list (format "charset=%s" charset)))
      )
    ;; Insert ct and its params. This is flexible enough for msg/extb
    (mew-header-insert-here mew-ct: ct 'noret)
    (if (null params)
	()
      (insert (format "; %s" (car params)))
      (setq params (cdr params))
      (while params
	(insert (format ";\n\t%s" (car params)))
	(setq params (cdr params))))
    (insert "\n")
    ;; Insert cte.
    (mew-header-insert-here mew-cte: cte) ;; cte is never nil here
    ;; Insert cd.
    (if cd (mew-header-insert-here 
	    mew-cd: 
	    (concat "\"" (mew-header-encode cd "B") "\"")))
    ;; Terminate content-header.
    (insert "\n")
    ;; header "\n" (cur) [text]
    (if encopts
	;; Binary objects matches here.
	;; If called charset-magic and encopts is not nil, file is not nil.
	;; Encoded file is always in us-ascii domain. So just insert it.
	(mew-plet
	 (apply (function call-process) mew-prog-mime-encode 
		file t nil encopts)
	 )
      (if file ;; not buffered	 
	  ;; Probably 7bit text based object like phantom body of 
	  ;; message/external-body. 
	  (insert-file-contents file)))
    ;; Go to RFC1847 privacy services.
    ;; RFC1847 and services using it such as RFC1848 MOSS and RFC2015 PGP/MIME
    ;; allows 8bit/binary MIME canonical form(i.e. 7bit content-header and 
    ;; 8bit/binary content-body) for encryption (not for signing). 
    ;; But Mew restricts an object to be ecnrypted(or, of course, signed) 
    ;; to be in 7bit for the following two reasons:
    ;;   (1) It is technically awkward that an object consists of 
    ;;      a 7bit header part and a BINARY body part. For the former
    ;;      is text based but the latter is not text based.
    ;;   (2) 8BIT TEXT objects are obstacles for multi-lingual environment.
    ;;      US-ASCII, 7bit-encoded 8bit text, ISO-2022 families are allowed.
    ;;      (It's not impossible to implement but too few advantage.)
    ;;
    ;; Now we gained 7bit-encoded MIME canonical object here.
    ;;         content-header<LF>
    ;;         <LF>
    ;;         7bit-content-body<LF>
    ;;
    (if (null privacy)
	()
      (save-restriction
	(narrow-to-region beg (point-max))
	(let ((case-fold-search t) 
	      (decrypters (mew-syntax-get-decrypters syntax))
	      protocol ct)
	  (while privacy
	    (goto-char (point-min)) 
	    (setq ct (nth 0 (car privacy)))
	    (setq protocol (nth 1 (car privacy)))
	    (setq privacy (cdr privacy))
	    (cond 
	     ((string-match mew-ct-mle ct)
	      (mew-encode-multipart-encrypted ct protocol depth decrypters))
	     ((string-match mew-ct-mls ct)
	      (mew-encode-multipart-signed ct protocol depth))
	     )
	    ))))
    (goto-char (point-max))
    ))

(defun mew-security-multipart-boundary (depth)
   (if depth
       (mew-boundary-get (format "Security_Multipart%s" (int-to-string depth)))
     (mew-boundary-get "Security_Multipart")))

(defun mew-draft-make-multi ()
  ;;
  ;; delete multipart syntax
  ;;
  (goto-char (marker-position mew-draft-buffer-attach))
  (delete-region (point) (point-max))
  ;; disable marker
  (set-marker mew-draft-buffer-attach nil)
  (setq mew-draft-buffer-attach nil)
  ;; back up the draft and its syntax
  (let* ((path (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	 (backup-file (concat path "/" mew-draft-backup-file))
	 (syntax-file (concat path "/" mew-draft-syntax-file))
	 (syntax mew-encode-syntax) ;; mew-encode-syntax is buffer local
	 )
    (write-region (point-min) (point-max) backup-file)
    (save-excursion
      (set-buffer (find-file-noselect syntax-file))
      (erase-buffer)
      (prin1 syntax (current-buffer)) ;; different buffer, so use syntax
      (terpri (current-buffer))
      (save-buffer)
      (message "") ;; flush echo area
      (kill-buffer (current-buffer))
      ))
  ;;
  ;; delete body
  ;;
  ;; delete delimiter
  (mew-draft-refresh)
  (goto-char (marker-position mew-draft-buffer-header))
  ;; the beginning of "----"
  (let ((beg (point))
	(syntax mew-encode-syntax) 
	;; use local variable to prevent side effects when undo
	(coverfile (concat 
		    (mew-expand-file-name (mew-draft-to-mime (buffer-name)))
		    "/" mew-draft-coverpage))
	(path (mew-expand-file-name mew-draft-mime-folder))
	)
    (forward-line)
    ;; cursor is just after "----\n"
    (delete-region beg (point))
    ;; disable marker
    (set-marker mew-draft-buffer-header nil)
    (setq mew-draft-buffer-header nil)
    ;; just after the header
    ;; see if cover page is empty or not
    (while (and (looking-at "^$") (not (eobp)))
      (forward-line))
    (if (not (eobp))
	;; A cover page exists, so save it.
	;; On Mule, from internal to mew-cs-draft.
	;; This code may be overhead but makes code simple.
	(mew-frwlet
	 mew-cs-noconv mew-cs-draft
	 (write-region beg (point-max) coverfile)
	 )
      ;; A cover page doesn't exist
      ;; Remove the cover page entry from the syntax
      (setq syntax (mew-syntax-remove-entry syntax '(1))))
    (delete-region beg (point-max))
    ;; Now only the header in the draft
    ;;
    ;; Let's go multi
    ;;
    (mew-encode-multipart syntax path 0)
    ))

(defun mew-encode-multipart (syntax path depth)
  (let* ((boundary
	  (mew-boundary-get ;; 0 is nil for Next_Part
	   (if (> depth 0) (format "BOUNDARY%s" (int-to-string depth)))))
	 (fullname (mew-expand-file-name (mew-syntax-get-file syntax) path))
	 (ctl (mew-syntax-get-ct syntax))
	 (ct (capitalize (car ctl))) ;; just in case
	 (cte (mew-syntax-get-cte syntax))
	 (cd (mew-syntax-get-cd syntax))
	 (privacy (mew-syntax-get-privacy syntax))
	 (len (length syntax))
	 (cnt mew-syntax-magic)
	 (beg (point)))
    (mew-header-insert-here 
     mew-ct: (concat ct ";\n\tboundary=\"" boundary "\""))
    (mew-header-insert-here mew-cte: (or cte "7bit"))
    (if cd (mew-header-insert-here 
	    mew-cd: 
	    (concat "\"" (mew-header-encode cd "B") "\"")))
    (while (< cnt len)
      (insert (concat "\n--" boundary "\n"))
      (if (mew-syntax-multipart-p (aref syntax cnt))
	  (mew-encode-multipart (aref syntax cnt) fullname (1+ depth))
	(mew-encode-singlepart (aref syntax cnt) fullname (1+ depth)))
      (setq cnt (1+ cnt))
      )
    (insert (concat "\n--" boundary "--\n"))
    ;; 
    ;; Let's go privacy services.
    ;;
    (if (null privacy)
	()
      (save-restriction
	(narrow-to-region beg (point-max))
	(let ((case-fold-search t) 
	      (decrypters (mew-syntax-get-decrypters syntax))
	      protocol ct)
	  (while privacy
	    (goto-char (point-min)) 
	    (setq ct (nth 0 (car privacy)))
	    (setq protocol (nth 1 (car privacy)))
	    (setq privacy (cdr privacy))
	    (cond 
	     ((string-match mew-ct-mle ct)
	      (mew-encode-multipart-encrypted ct protocol depth decrypters))
	     ((string-match mew-ct-mls ct)
	      (mew-encode-multipart-signed ct protocol depth))
	     )
	    ))))
    (goto-char (point-max))
    ))

;;;
;;; Privacy services
;;;

(defun mew-encode-multipart-encrypted (ct protocol depth decrypters)
  ;; called in the narrowed region
  (let* ((boundary (mew-security-multipart-boundary depth))
	 (switch mew-encode-multipart-encrypted-switch) ;; save length
	 (func (cdr (mew-assoc protocol switch 0 t)))
	file1 file2 file3 cte2 cte3 fc)
    (setq decrypters (mew-encode-decrypters-list decrypters))
    ;; Write the part converting line breaks.
    (setq file1 (mew-save-transfer-form (point-min) (point-max)))
    ;; The narrowed region stores nothing
    ;; Call the protocol function
    (setq fc (funcall func file1 decrypters))
    (setq file2 (nth 0 fc) cte2 (nth 1 fc) file3 (nth 2 fc) cte3 (nth 3 fc))
    ;; Create multipart content-header
    (mew-header-insert-here mew-ct: ct 'noret)
    (insert (format ";\n\tprotocol=\"%s\"" protocol))
    (insert (format ";\n\tboundary=\"%s\"\n" boundary)) ;; \n doubled
    (insert (format "\n--%s\n" boundary))
    ;; Insert control keys
    (mew-encode-singlepart 
     (mew-encode-syntax-single file2 (list protocol) cte2) nil nil)
    (insert (format "\n--%s\n" boundary))
    ;; Insert encrpted body
    (mew-encode-singlepart 
     (mew-encode-syntax-single file3 mew-type-apo cte3) nil nil)
    (insert (format "\n--%s--\n" boundary))
    ;; Throw away the garbage 
    (if (file-exists-p file1) (delete-file file1))
    (if (file-exists-p file2) (delete-file file2))
    (if (file-exists-p file3) (delete-file file3))
    )
  )

(defun mew-encode-multipart-signed (ct protocol depth)
  ;; called in the narrowed region
  (let* ((boundary (mew-security-multipart-boundary depth))
	 (switch mew-encode-multipart-signed-switch);; save length
	 (func (cdr (mew-assoc protocol switch 0 t)))
	 file1 file2 micalg cte2 fmc)
    (setq file1 (mew-save-signature-form (point-min) (point-max)))
    ;; The narrowed region still the ORIGINAL part (i.e. line breaks are LF)
    ;; Call the protocol function
    (setq fmc (funcall func file1))
    (setq file2 (nth 0 fmc) cte2 (nth 1 fmc) micalg (nth 2 fmc))
    (goto-char (point-min))
    ;; Before the signed part
    ;; Create multipart content-header
    (mew-header-insert-here mew-ct: ct 'noret)
    (insert (format ";\n\tprotocol=\"%s\"" protocol))
    (insert (format ";\n\tmicalg=\"%s\"" micalg))
    (insert (format ";\n\tboundary=\"%s\"\n" boundary)) ;; \n doubled
    (insert (format "\n--%s\n" boundary))
    (goto-char (point-max))
    ;; After the sigend part
    (insert (format "\n--%s\n" boundary))
    (mew-encode-singlepart 
     (mew-encode-syntax-single file2 (list protocol) cte2) nil nil)
    (insert (format "\n--%s--\n" boundary))
    ;; Throw away the garbage 
    (if (file-exists-p file1) (delete-file file1))
    (if (file-exists-p file2) (delete-file file2))
    )
  )

(defun mew-save-transfer-form (beg end)
  ;; called in the narrowed region
  (mew-frwlet 
   mew-cs-noconv mew-cs-7bit-crlf
   (let (file)
     (if mew-cs-7bit-crlf
	 ()
       (goto-char (point-min)) ;; just in case
       (while (search-forward "\n" nil t) (replace-match "\r\n" nil t))
       (setq end (point-max))
       )
     (if (not (file-exists-p mew-temp-dir))
	 (mew-make-directory mew-temp-dir)) ;; just in case
     (setq file (make-temp-name mew-temp-file))
     (write-region beg end file)
     (delete-region beg end)
     file ;; return value
     )
   ))

(defun mew-save-signature-form (beg end)
  (mew-frwlet
   mew-cs-noconv mew-cs-7bit
   (let (file1 file2)
     (save-excursion 
       (save-restriction
	 (narrow-to-region beg end)
	 ;; NEVER use call-process-region for privary reasons
	 (if (not (file-exists-p mew-temp-dir))
	     (mew-make-directory mew-temp-dir)) ;; just in case
	 (setq file1 (make-temp-name mew-temp-file))
	 (write-region (point-min) (point-max) file1)
	 (setq file2 (make-temp-name mew-temp-file))
	 (call-process "mewencode" file1 nil nil "-s" "-" file2)
	 (if (file-exists-p file1) (delete-file file1))
	 file2 ;; return value
	 ))
     ))
  )

(defun mew-encode-decrypters-list (string)
  (let ((list))
    (setq list (mew-header-canform-list
                (mew-header-expand-alias-list
                 (mew-header-delete-nullstring-list
                  (mew-header-extract-addr-list
                   (mew-header-split
		    (mew-header-syntax string)
                    ?,))))))
    (cons mew-mail-address list);; to decrypt by myself
    ))

(defvar mew-default-boundary "--%s(%s)--")

(defun mew-boundary-get (&optional string)
  (if (null string) (setq string "Next_Part"))
  (format mew-default-boundary
	  string
	  (mew-replace-character (current-time-string) 32 ?_) ; 32 == " "
	  )
  )

;;
;;
;;

(defun mew-charset-8bit-p (charset)
  (let ((case-fold-search t))
    (cond 
     ((string-match "iso-8859" charset) t)
     ((string-match "iso-2022" charset) nil)
     ((string-match "us-ascii" charset) nil)
     (t t) ;; just in case
     )
    ))

(defun mew-charset-iso-2022-p (charset)
  (let ((case-fold-search t))
    (cond 
     ((string-match "iso-2022" charset) t)
     (t nil)
     )
    ))


(provide 'mew-encode)
