;;;			     mew-draft.el
;;;
;;;	      Copyright (C) 1996-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: October  2, 1996
;;; Revised: April   21, 1997
;;;

(defconst mew-draft-version "mew-draft.el version 0.15")

(require 'mew)

(defvar mew-draft-mode-map   nil)
(defvar mew-draft-header-map nil)
(defvar mew-draft-body-map   nil)

(defvar mew-draft-mode-syntax-table nil
  "*Syntax table used while in mew letter mode.")

(defvar mew-address-separator ":, \t\n")

(defvar mew-fields
  '("To:" "Cc:" "Subject:" "Dcc:" "Fcc:" "Bcc:"
    "Reply-To:" "Followup-To:" "From:" "Newsgroups:")
  "*Completion field list on draft mode"
  )

(defvar mew-field-completion-switch
  '(("To:" . mew-draft-address-comp)
    ("Cc:" . mew-draft-address-comp)
    ("Dcc:" . mew-draft-address-comp)
    ("Bcc:" . mew-draft-address-comp)
    ("Reply-To:" . mew-draft-address-comp)
    ("From:"     . mew-draft-address-comp)
    ("Fcc:" . mew-draft-folder-comp))
  "*Completion function alist concerned with the key"
  )

(if mew-draft-mode-syntax-table
    ()
  (setq mew-draft-mode-syntax-table
	(make-syntax-table text-mode-syntax-table))
  (set-syntax-table mew-draft-mode-syntax-table)
  (modify-syntax-entry ?% "." mew-draft-mode-syntax-table)
  )

(defvar mew-draft-mode-toolbar-menu
  '("Attachment Commands"
    ["Insert a File by Linking"
     mew-attach-link
     (mew-attach-not-line012-1)]
    ["Insert a File by Copying"
     mew-attach-copy
     (mew-attach-not-line012-1)]
    ["Insert Audio"
     mew-attach-audio
     (mew-attach-not-line012-1)]
    ["Insert an External Reference"
     mew-attach-external-body
     (mew-attach-not-line012-1)]
    ["Insert a Sub-Multipart"
     mew-attach-multipart
     (mew-attach-not-line012-1)] 
    ["Read a New File into a Buffer"
     mew-attach-find-mew-file
     (mew-attach-not-line012-1)]
    ["Insert PGP public keys"
     mew-attach-pgp-public-key
     (mew-attach-not-line012-1)]
    "----"
    ["Delete This Part"
     mew-attach-delete 
     (mew-attach-not-line012-1-dot)]
    "----"
    ["Describe This Part"
     mew-attach-description
     (mew-attach-not-line0-1-dot)]
    ["Change the Type"
     mew-attach-type
     (mew-attach-not-line0-1-dot)]
    ["Encode with Gzip64"
     mew-attach-gzip64
     (mew-attach-not-line0-1-dot)]
    ["Encode with Base64"
     mew-attach-base64
     (mew-attach-not-line0-1-dot)]	 
    ["Encode with Quoted-Printable"
     mew-attach-quoted-printable
     (mew-attach-not-line0-1-dot)]
    ["Sign with PGP"
     mew-attach-pgp-sign
     (mew-attach-not-line0-1-dot)]
    ["Encrypt with PGP"
     mew-attach-pgp-enc 
     (mew-attach-not-line0-1-dot)]
    "----"
    ["Undo Encoding"
     mew-attach-undo
     (mew-attach-not-line0-1-dot)]
    "----"
    ["Read This File into a Buffer"
     mew-attach-find-file
     (mew-attach-not-line012-1-dot)]
    )
  )

(defvar mew-draft-mode-menu-spec
  (list
   "Mew/draft"
   ["Cite" mew-draft-cite t]
   ["Cite without Label" mew-draft-yank t]
   mew-draft-mode-toolbar-menu
   ["Make MIME Message"   mew-draft-make-mime           t]
   ["Send Message"        mew-draft-send-letter         t]
   ["Prepare Attachments" mew-draft-prepare-attachments t]
   ["Insert Signature"    mew-draft-insert-signature    t]
   ["Kill Draft"          mew-draft-kill                t]
   "----"
   '("PGP"
     ["PGP Sign"              mew-pgp-sign-letter         (not (mew-attach-p))]
     ["PGP Encrypt"           mew-pgp-encrypt-letter      (not (mew-attach-p))]
     ["PGP Sign then Encrypt" mew-pgp-sign-encrypt-letter (not (mew-attach-p))]
     )
   '("FIB"
     ["FIB next item"     mew-fib-next-item     (not (mew-attach-p))]
     ["FIB previous item" mew-fib-previous-item (not (mew-attach-p))]
     ["FIB flush input"   mew-fib-flush-input   (not (mew-attach-p))]
     ["FIB fill default"  mew-fib-fill-default  (not (mew-attach-p))]
     ["FIB delete frame"  mew-fib-delete-frame  (not (mew-attach-p))]
     )
    )
  )

(if mew-draft-mode-map
    ()
  (setq mew-draft-mode-map (make-sparse-keymap))
  (let ((begin ?\40) (end ?\177))
    (while (<= begin end)
      (define-key mew-draft-mode-map 
	(char-to-string begin) 'mew-draft-keyswitch)
      (setq begin (1+ begin))))
  (define-key mew-draft-mode-map "\C-n"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-p"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-f"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-b"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-l"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\t"       'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-c\t"   'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-d"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-o"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-q"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-t"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-w"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-k"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\r"       'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\n"       'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-y"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-c\C-y" 'mew-draft-cite)
  (define-key mew-draft-mode-map "\C-cy"    'mew-draft-yank)
  (define-key mew-draft-mode-map "\C-c\C-w" 'mew-draft-check-whom)
  (define-key mew-draft-mode-map "\C-c\C-m" 'mew-draft-make-mime)
  (define-key mew-draft-mode-map "\C-c\C-u" 'mew-draft-undo)
  (define-key mew-draft-mode-map "\C-c\C-c" 'mew-draft-send-letter)
  (define-key mew-draft-mode-map "\C-c\C-s" 'mew-pgp-sign-letter)
  (define-key mew-draft-mode-map "\C-c\C-e" 'mew-pgp-encrypt-letter)
  (define-key mew-draft-mode-map "\C-c\C-b" 'mew-pgp-sign-encrypt-letter)
  (define-key mew-draft-mode-map "\C-c\C-q" 'mew-draft-kill)
  (define-key mew-draft-mode-map "\C-cM"    'mew-draft-prepare-attachments)
  (define-key mew-draft-mode-map "\C-c\C-n" 'mew-fib-next-item)
  (define-key mew-draft-mode-map "\C-c\C-p" 'mew-fib-previous-item)
  (define-key mew-draft-mode-map "\C-cu"    'mew-fib-flush-input)
  (define-key mew-draft-mode-map "\C-c\C-f" 'mew-fib-fill-default)
  (define-key mew-draft-mode-map "\C-c\C-k" 'mew-fib-delete-frame)
  (if mew-emacs-p
      (easy-menu-define
       mew-draft-mode-menu
       mew-draft-mode-map
       "Menu used in Mew draft mode."
       mew-draft-mode-menu-spec))
  )

(if mew-draft-header-map
    ()
  (setq mew-draft-header-map (make-sparse-keymap))
  (define-key mew-draft-header-map "\t"     'mew-draft-header-comp)
  (define-key mew-draft-header-map "\C-c\t" 'mew-draft-header-domain)
  (if mew-use-bbdb
      (define-key mew-draft-header-map "\e\t" 'bbdb-complete-name))
  )

(if mew-draft-body-map
    ()
  (setq mew-draft-body-map (make-sparse-keymap))
  (define-key mew-draft-body-map "\t"     'tab-to-tab-stop)
  (define-key mew-draft-body-map "\C-c\t" 'mew-draft-insert-signature)
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Draft mode
;;;

(defvar mew-draft-window-config nil)

(defun mew-draft-mode ()
  "Major mode for composing a MIME message.
Key actions are different in each region: Header, Body, Attachment.

To send a draft, type C-cC-m and C-cC-c.
To make multipart, type C-cM, edit attachments, and type C-cC-m and C-cC-c.

*Whole buffer key assignment:

C-cC-y	  Cite from *mew message* buffer.
C-uC-cC-y Cite both the header and the body with a label and prefix.
C-cy	  Yank the body.

C-cC-m	Make a MIME message. Charset guess, mapping directory structure 
	to multipart, and so on.
C-cC-c	Send this message. If you skipped C-cC-m, Content-Type: is added
	to header and you are asked, 
	\"Content-Type: was automatically added. Send this mail? \".
	Type y or n. 
	Mew sends the message in background. So, when you exit Emacs, you
	may be asked, 
	\"Active processes exist; kill them and exit anyway? (yes or no)\".
	In this case, check if *mew watch* buffer exist. If so, never 
	exit Emacs because Mew is still sending the message.

C-cC-u	Undo C-cC-m on multipart.
C-cC-q	Kill this draft.

C-cC-e	Encrypt this message by PGP.
C-cC-s	Sign this message by PGP.
C-cC-b	Sign then encrypt this message by PGP.

C-cM	Prepare attachments region.

*Header region key assignment:

TAB	Complete field keys.
	Complete and expand email address aliases.
	Complete folder names.
C-cTAB	Complete your mail domain.


*Body region key assignment:

C-cTAB	Insert signature file.


*Attachments region Key assignment:

C-f	Go to the first subdirectory.
C-b	Go to the parent directory.
C-p	Go to the previous file in the current directory.
C-n	Go to the next file in the current directory.

c	Copy a file (via networks) on \".\".
l	Link a file with a symbolic link on \".\".
d	Delete this file or this directory.
m	Create a subdirectory(i.e. multipart) on \".\".
f	Open this file into a buffer.
F	Open a new file into a buffer on \".\".
e	Input external-body on \".\".
a	Sampling voice and insert as audio file on \".\".
p	Extract the PGP key for the inputed user on \".\".
D	Input a description(Content-Description:).
T	Change the data type(Content-Type:).
C	Specify charset for a Text/* object.

B	Mark as Base64(B).
Q	Mark as Quoted-Printable(Q).
G	Mark as Gzip64(G).
S	Mark as PGP sign(PS).
E	Mark as PGP encrypt. Input to whom you encrypt this part(PE).
U	Get mark back.


* Fill blanks

Prepare ~/.mew-fib like;

	name:  Kazuhiko Yamamoto
	email: kazu@is.aist-nara.ac.jp

If you receive a message like;

	Your name : |>name<|
	Your e-mail address: |>email<|

Type a (in summary mode), C-cy, C-cC-f, and C-cC-k makes following
draft.

	Your name : Kazuhiko Yamamoto
	Your e-mail address: kazu@is.aist-nara.ac.jp

In this way, mew-fil fills up items quoted like |> <| from .mew-fib.

C-cy	Yank from *mew message* buffer without citation label.
C-cC-f	Fill |>item<| from .mew-fib.
C-cC-k	Delete all quotations, i.e. |> <|.
C-cC-n	Jump to the next fib item.
C-cC-p	Jump to the previous fib item.
C-cu	Flush input from .mew-fib.

Moreover, mew-fib supports aliases like;

	email: kazu@is.aist-nara.ac.jp
	e-mail:

"
  (interactive)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^[ \t]*[-_][-_][-_]+$\\|" paragraph-start))
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate
        (concat "^[ \t]*[-_][-_][-_]+$\\|" paragraph-separate))
  (setq major-mode 'mew-draft-mode)
  (setq mode-name "Draft")
  (use-local-map mew-draft-mode-map)
  (set-syntax-table mew-draft-mode-syntax-table)
  (cd (expand-file-name "~"))
  (run-hooks 'text-mode-hook 'mew-draft-mode-hook)
  (if mew-xemacs-p
      (let ((highlight-headers-hack-x-face-p nil))
	;; there may not be From: field
	(highlight-headers 
	 (point-min) (marker-position mew-draft-buffer-header) nil)
	(set-buffer-menubar current-menubar)
	(add-submenu nil mew-draft-mode-menu-spec)
	))
  (if mew-icon-p
      (set-specifier default-toolbar
		     (cons (current-buffer) mew-draft-toolbar)))
  ;; to seed up...
  (if (featurep 'hilit19)
      (hilit-rehighlight-buffer)
    (redraw-display))
  (cond 
   ((equal mew-mule-version 2)
    (make-local-variable 'file-coding-system-for-read)
    (make-local-variable 'file-coding-system)
    (setq file-coding-system-for-read mew-cs-autoconv)
    (setq file-coding-system          mew-cs-draft))
   ((equal mew-mule-version 3)
    (setq coding-system-for-read  mew-cs-autoconv)
    (setq coding-system-for-write mew-cs-draft))
   )
  (if (and auto-fill-function mew-emacs-p)
      (progn
	(make-local-variable 'auto-fill-function)
	(setq auto-fill-function (function mew-draft-auto-fill))
	))
  )

(defun mew-draft-auto-fill ()
  (do-auto-fill)
  (if (mew-draft-in-header-p)
      (save-excursion
	(beginning-of-line)
	(if (not (looking-at "[^ \t]+:\\|[ \t]"))
	    (insert "\t"))
	)))

(defun mew-draft-get-new (&optional folder)
  (save-excursion
    (set-buffer mew-buffer-tmp)
    (erase-buffer)
    (call-process "mhpath" nil (current-buffer) nil
		   (or folder mew-draft-folder)
		   "new")
    (goto-char (point-min))
    (forward-line)
    (mew-buffer-substring (point-min) (1- (point)))
    ))

(defun mew-draft-rename (file)
  (rename-buffer (concat mew-draft-folder 
			 "/" 
			 (file-name-nondirectory file)))
  )

(defmacro mew-to-cc-magic ()
  (` (or (catch (quote match)
	   (car (mapcar
		 (function (lambda (arg)
			     (and (string-match arg c)
				  (throw (quote match) t))))
		     del)))
	 (string-match ":;" c))))

(defun mew-draft-header (&optional subject nl to cc newsgroups in-reply-to references)
;; to -- string or list
;; cc -- string or list
;; nl -- one empty line under "----", which is necessary if
;;      multipart region is prepared
  (let ((del (cons (concat "^" (regexp-quote (user-login-name)) "$")
		   (cons (concat "^" mew-mail-address "$")
			 mew-mail-address-list)))
	;; deleting list for Cc:
	(c nil)
	(cep nil)
	(tf to)) ;; to is used in last
    (goto-char (point-min))
    ;; Insert To: at first.
    ;; All addresses inserted on To: are appended to del.
    (cond
     ((null tf) (insert "To: \n"))
     ;; To: inputed from the mini-buffer.
     ((stringp tf)
      ;; Cc: is also string
      ;; We believe that user never specifies the same address of To: to Cc:
      ;; So, not add to to del.
      (mew-header-insert-here "To:" tf)) 
     ;; To: collected by reply
     ((listp tf)
      (insert (format "To: %s" (car tf)))
      (setq del (cons (concat "^" (regexp-quote (car tf)) "$") del))
      (while tf
	(setq c (car tf) tf (cdr tf))
	(if (mew-to-cc-magic)
	    ()
	  (insert (format ",\n\t%s" c))
	  (setq del (cons (concat "^" (regexp-quote c) "$") del))
	  ))
      (insert "\n"))
     )
    (cond
     ((null cc) ()) ; do nothing 
     ;; Cc: inputed from the mini-buffer.
     ((stringp cc)
      (mew-header-insert-here "Cc:" cc))
     ;; Cc: collected by reply.
     ((listp cc)
      ;; find the first Cc: value since this is preceding by Cc:
      (catch 'first
	(while cc
	  (setq c (car cc) cc (cdr cc))
	  (if (mew-to-cc-magic)
	      ()
	    (insert "Cc: ")
	    (insert c)
	    (setq del (cons (concat "^" (regexp-quote c) "$") del))
	    (setq cep t)
	    (throw 'first nil))
	  ))
      ;; insert the second or more Cc: preceding ",\n\t"
      (while cc
	(setq c (car cc) cc (cdr cc))
	(if (mew-to-cc-magic)
	    ()
	  (insert (format ",\n\t%s" c))
	  (setq del (cons (concat "^" (regexp-quote c) "$") del))
	  ))
	(if cep (insert "\n")))
     ))
  (and newsgroups (mew-header-insert-here "Newsgroups:" newsgroups))
  (and mew-cc   (mew-header-insert-here "Cc:" mew-cc))
  (mew-header-insert-here "Subject:" (if subject subject "")) ;; tricky
  (and mew-from (mew-header-insert-here "From:" mew-from))
  (and mew-fcc  (mew-header-insert-here "Fcc:" mew-fcc))
  (and mew-dcc  (mew-header-insert-here "Dcc:" mew-dcc))
  (and mew-reply-to (mew-header-insert-here "Reply-To:" mew-reply-to))
  ;; xxx date to in-reply-to ?
  (and in-reply-to (mew-header-insert-here 
		    "In-Reply-To:"
		    (concat "Your message of \"" in-reply-to "\"")))
  (and references (mew-header-insert-here "References:" references))
  (if (and mew-x-face-file
	   (file-exists-p (expand-file-name mew-x-face-file)))
      (let ((xface))
	(save-excursion
	  (set-buffer mew-buffer-tmp)
	  (erase-buffer)
	  (insert-file-contents (expand-file-name mew-x-face-file))
	  (setq xface (mew-buffer-substring (point-min) (max (buffer-size) 1)))
	  )
	(mew-header-insert-here "X-Face:" xface)
	))
  (and mew-x-mailer (mew-header-insert-here "X-Mailer:" mew-x-mailer))
  (let ((halist mew-header-alist))
    (while halist
      (and (stringp (car (car halist))) (stringp (cdr (car halist)))
	   (mew-header-insert-here (car (car halist)) (cdr (car halist))))
      (setq halist (cdr halist)))
    )
  (mew-header-insert-here "Mime-Version:" mew-mime-version)
  (insert mew-header-separator "\n")
  (mew-draft-refresh)
  (if nl (insert "\n"))
  (if to ;; there is no To:
      ()
    (goto-char (point-min))
    (end-of-line))
  )

(defun mew-draft-refresh ()
  "Refresh draft buffer"
  (interactive)
  (save-excursion
    (widen)
    (goto-char (point-min))
    (re-search-forward mew-eoh2)
    (beginning-of-line)
    (setq mew-draft-buffer-header (point-marker))
    )
  )

(defun mew-draft-send-letter (&optional arg)
  "Send this message. If arg is specified, leave the draft as it is."
  (interactive "P")
  (mew-draft-refresh)
  (if mew-ask-subject
      (save-excursion
	(goto-char (point-min))
	(re-search-forward "^\\(Subject:.*\\|-+\\)$" nil t 1)
	(if (string-match "^Subject: *$"
			  (buffer-substring (match-beginning 0) (match-end 0)))
	    (insert (read-from-minibuffer "Subject: " nil nil)))))
  (if mew-ask-newsgroups
      (let ((existp nil))
	(save-restriction
	  (narrow-to-region (point-min)
			    (marker-position mew-draft-buffer-header))
	  (save-excursion
	    (goto-char (point-min))
	    (if (re-search-forward "^Newsgroups:" nil t)
		(setq existp t))
	    ))
	(if (and existp 
		 (not (mew-y-or-n-p "Do you want to post to NetNews ?")))
	    (mew-header-delete-lines '("Newsgroups:")))
	))
  (run-hooks 'mew-send-hook)
  (save-excursion
    (if (mew-header-get-value mew-ct:)
	(mew-draft-real-send-letter arg)
      (message "Making a MIME letter ...")
      (mew-draft-make-mime)
      (message "Making a MIME letter ... done")
      (condition-case nil
	  (if (or mew-auto-add-content-type
		  (mew-y-or-n-p
		   (format "%s was automatically added. Send this mail? "
			   mew-ct:)))
	      (mew-draft-real-send-letter arg)
	    (mew-draft-delete-content-type))
	(quit
	 (mew-draft-delete-content-type)))
      )
    ))

(defun mew-draft-delete-content-type ()
  (mew-header-delete-lines (list mew-ct: mew-cte:))
  (mew-draft-refresh)
  (save-excursion
    (goto-char (marker-position mew-draft-buffer-header))
    (insert mew-header-separator)
  ))

(defun mew-draft-real-send-letter (&optional arg)
  (let* ((mimefolder (mew-draft-to-mime (buffer-name)))
	 (mimedir (mew-expand-file-name mimefolder))
	 (msg (file-name-nondirectory (buffer-file-name)))
	 (fcc nil)
	 (process-connection-type mew-connection-type1)
	 (delete nil))
    (set-buffer-modified-p t)		; Make sure buffer is written
    (mew-frwlet 
     mew-cs-noconv mew-cs-mime-trans
     (save-buffer)
     )
    ;; make backup folder(s)
    ;; Fcc: fields may have multiple values.
    (setq fcc (mew-header-address-collect '("Fcc:")))
    (while fcc
      (if (file-exists-p (mew-expand-file-name (car fcc)))
	  () ;; do nothing
	(call-process mew-prog-folder nil nil nil
		      "-create"
		      (mew-dir-to-folder (car fcc))))
      (setq fcc (cdr fcc)))
    (if arg
	() ;; leave the draft
      (kill-buffer (current-buffer))
      ;;
      (if (mew-current-get 'window)
	  (progn
	    (set-window-configuration (mew-current-get 'window))
	    (mew-current-set 'window nil))))
    (set-buffer (generate-new-buffer mew-buffer-watch))
    ;; watch buffer
    (setq mew-watch-buffer-process
	  (start-process "Send" (current-buffer)
			 mew-prog-send
			 "-draftfolder" mew-draft-folder
			 "-draftmessage" msg
			 "-watch" "-verbose"))
    (mew-set-process-cs mew-watch-buffer-process
			mew-cs-noconv mew-cs-mime-trans)
    (set-process-sentinel mew-watch-buffer-process
			  'mew-watch-sentinel)
    (message "Sending a message in background")
    (if (null (file-directory-p mimedir))
	()
      (cond
       ((equal mew-mime-compose-folder-delete 'ask)
	(setq delete (mew-y-or-n-p
		      (format "Folder %s exists. Remove it? " mimefolder))))
       ((equal mew-mime-compose-folder-delete 'delete)
	(setq delete t))
       ((equal mew-mime-compose-folder-delete 'retain)
	(setq delete nil))
       (t
	(setq delete (mew-y-or-n-p
		      (format "Folder %s exists. Remove it? " mimefolder))))
       )
      (if (null delete )
	  (format "Folder %s remains" mimefolder)
	(message "Removing folder %s ... " mimefolder)
	(mew-delete-directory-recursively mimedir)
	(message "Removing folder %s ... done" mimefolder))
      )
    ))

(defun mew-watch-sentinel (process event)
  (let ((cbuf (current-buffer)) (kbuf (process-buffer process)))
    (set-buffer kbuf)
    (goto-char (point-min))
    (if (null (re-search-forward "^send:" nil t))
	(progn
	  (set-buffer cbuf)  ;; to avoid cursor-in-echo-area bug
	  (kill-buffer kbuf) ;; set-buffer before kill-buffer
	  )
      (ding)
      (message "Send failed")
      (beginning-of-line)
      (switch-to-buffer (process-buffer process))
      (local-set-key "\C-c\C-q" 'mew-kill-buffer)
      )
    ))

(defun mew-draft-yank ()
  (interactive)
  (mew-draft-refresh)
  (let ((cite))
    (save-excursion
      (set-buffer (mew-buffer-message))
      (save-restriction
	(widen)
	(setq cite
	      (cond
	       ((if mew-xemacs-p 
		    (mark t)
		  (marker-position (mark-marker))) ;;; (mark)
		(mew-buffer-substring (region-beginning) (region-end))
		)
	       ((equal mew-message-citation 'noheader)
		(mew-buffer-substring (point-min) (point-max)))
	       (t
		(goto-char (marker-position mew-draft-buffer-header))
		(forward-line)
		(mew-buffer-substring (point) (point-max)))
	       )
	      )
	))
    (insert-string cite)
    ))

(defun mew-cite-strings ()
  (let ((fields (mapcar (function mew-header-get-value) mew-cite-fields)))
    (setq fields (mapcar (function (lambda (x) (if (null x) "" x))) fields))
    (apply (function format) mew-cite-format fields)
    ))

(defun mew-draft-cite (&optional arg)
  (interactive "P")
  (let ((cite "")
	(noheader (eq mew-message-citation 'noheader))
        (nonmewbuf mew-message-citation-buffer)) ; buffer local
    (save-excursion
      (set-buffer (or nonmewbuf (mew-buffer-message)))
      (save-restriction
        ;; first prepare "cite"
        (widen)
        (setq cite
	      (cond
	       ((if mew-xemacs-p 
		    (mark t)
		  (marker-position (mark-marker))) ;;; (mark)
		(setq noheader t) ;; -> cite
		(mew-buffer-substring (region-beginning) (region-end))
		)
	       (t ;; with header
                (mew-buffer-substring (point-min) (point-max))
		)
	       ))
        (if noheader
            (progn
              (set-buffer (or nonmewbuf
                              ;; header exists only in cache if multipart
                              (mew-cache-hit (mew-current-get 'message))
                              (mew-buffer-message)))
              (goto-char (point-min))
              (re-search-forward mew-eoh nil t)
              (setq cite (concat (mew-buffer-substring (point-min) (point))
                                 "\n" cite))
              ))
        ))
    (save-restriction
      (narrow-to-region (point)(point)) ;; for (goto-char (point-min))
      (insert cite)
      (push-mark (point) t t)
      (goto-char (point-min)))
    (cond
     (mew-cite-hook
      (run-hooks 'mew-cite-hook))
     (t (mew-cite-original arg))
     )
    ))

(defun mew-cite-original (&optional arg)
  (if (< (marker-position (mark-marker)) (point))
      (exchange-point-and-mark))
  (let ((beg (point)) (end (marker-position (mark-marker)))
        label prefix)
    (save-restriction
      (narrow-to-region beg end)
      (condition-case nil
          (setq label (mew-cite-strings))
        (error "Syntax of mew-cite-format was changed. Read explanation of mew-cite-fields"))
      (if (null mew-cite-prefix-function)
          (setq prefix mew-cite-prefix)
        (setq prefix (funcall mew-cite-prefix-function)))
      (if mew-cite-prefix-confirmp
          (let ((ask (read-string 
                      (format "Prefix (\"%s\"): " prefix) "")))
            (if (not (string= ask "")) (setq prefix ask))))
      ;; C-u C-c C-y cites body with header.
      (if (eq arg nil) 
	  ;; header has been already cited. So, delete it.
	  (delete-region beg
			 (progn
			   (re-search-forward mew-eoh2 nil t)
			   (forward-line)
			   (point))))
      (insert label)
      (set-mark (point)) ;; for C-x C-x
      (and (bolp) (insert prefix))
      (while (equal 0 (forward-line))
	(or (equal (point) (point-max))
	    (insert prefix)))
      ) ;; restriction
    )
  )

(defun mew-draft-check-whom ()
  "Verify recipients of the draft."
   (interactive)
  (let* ((to-cc (mew-header-address-collect
		'("To:" "Cc:" "Bcc:" "Apparently-To:")))
	 (exp-to-cc (mew-header-expand-alias-list to-cc))
	 (buf (current-buffer)))
    (message "Checking recipients ... ")
    (get-buffer-create mew-buffer-whom)
    (switch-to-buffer-other-window mew-buffer-whom)
    (erase-buffer)
    (mapcar
     (function (lambda (x) (insert (format "%s\n" x))))
     exp-to-cc
     )
    (pop-to-buffer buf)
    (message "Checking recipients ... done")
    ))

(defun mew-draft-kill ()
  (interactive)
  (let* ((mimefolder (mew-draft-to-mime (buffer-name))) ;; buffer will kill
	 (mimedir (mew-expand-file-name mimefolder))
	 (delete nil))
    (if (mew-y-or-n-p "Kill draft message? ")
	(progn
	  (set-buffer-modified-p t) ;; for delete recovery file
	  (save-buffer)
	  (if (file-exists-p (buffer-file-name))
	      (delete-file (buffer-file-name)))
	  (set-buffer-modified-p nil)
	  (and (get-buffer (mew-draft-to-mime (buffer-name)))
	       (kill-buffer (mew-draft-to-mime (buffer-name))))
	  (kill-buffer (buffer-name))
	  (if (mew-current-get 'window)
	      (progn
		(set-window-configuration (mew-current-get 'window))
		(mew-current-set 'window nil)))
	  (message "Draft was killed")
	  (if (null (file-directory-p mimedir))
	      ()
	    (cond
	     ((equal mew-mime-compose-folder-delete 'ask)
	      (setq delete (mew-y-or-n-p
			    (format "Folder %s exists. Remove it? "
				    mimefolder))))
	     ((equal mew-mime-compose-folder-delete 'delete)
	      (setq delete t))
	     ((equal mew-mime-compose-folder-delete 'retain)
	      (setq delete nil))
	     (t
	      (setq delete (mew-y-or-n-p
			    (format "Folder %s exists. Remove it? "
				    mimefolder))))
	     )
	    (if (null delete )
		(format "Folder %s remains" mimefolder)
	      (message "Removing folder %s ... " mimefolder)
	      (mew-delete-directory-recursively mimedir)
	      (message "Removing folder %s ... done" mimefolder))
	    )
	  )
      (message "Draft was not killed"))
    ))

(defun mew-draft-insert-signature ()
  (interactive)
  (let ((sigfile (expand-file-name mew-signature-file)))
    (if (not (file-exists-p sigfile))
	(message "No signature file %s" sigfile)
      (if (and (mew-attach-p) mew-signature-as-lastpart)
	  (progn
	    (goto-char (point-max))
	    (forward-line -2)
	    (mew-attach-forward)
	    (mew-attach-link sigfile "Signature")
	    (mew-attach-description mew-signature-description)
	    )
	(if mew-signature-insert-last 
	    (if (null (mew-attach-p))
		(goto-char (point-max))
	      (goto-char (marker-position mew-draft-buffer-attach))
	      (insert "\n")
	      (forward-char -1)))
	(insert-file-contents sigfile)
	))
    ))

;;;
;;; Draft magic functions
;;;

(defun mew-draft-in-header-p ()
  (if (markerp mew-draft-buffer-header)
      (<= (point) (marker-position mew-draft-buffer-header))
    nil)
  )

(defun mew-draft-in-attach-p ()
  (if (markerp mew-draft-buffer-attach)
      (>= (point) (marker-position mew-draft-buffer-attach))
    nil)
  )

(defun mew-draft-keyswitch ()
  (interactive)
  (let* ((key (this-command-keys))
	 (command (lookup-key (current-global-map) key))
	 (i 0) len func)
    (if (numberp command)
	(setq len command
	      command (lookup-key (current-global-map) 
				  (mew-subsequence key 0 len))
	      key (mew-subsequence key len)))
    (setq len (length key))
    (if (or (eq command 'universal-argument) (eq command 'digit-argument))
	(catch 'keyswitch
	  (while (and (eq command 'universal-argument)
		      (<= ?0 (aref key i)) (>= ?9 (aref key i)))
	    (setq i (1+ i)))
	  (while (< i len)
	    (if (eq 'mew-draft-keyswitch
		    (key-binding (char-to-string (aref key i))))
		(throw 'keyswitch (if (vectorp key)
				      (setq key (mew-subvector key i))
				    (setq key (substring key i)))))
	    (setq i (1+ i))
	    )
	  ))
    (cond
     ((and (markerp mew-draft-buffer-attach) (mew-draft-in-attach-p))
      (setq func (lookup-key mew-draft-attach-map key)))
     ((and (markerp mew-draft-buffer-header) (mew-draft-in-header-p))
      (setq func (lookup-key mew-draft-header-map key)))
     (t 
      (setq func (lookup-key mew-draft-body-map key))
      )
     )
    (if (not (integerp func))
	()
      (if (vectorp key) (setq key (mew-subvector key 0 func))
	(setq key (substring key 0 func)))
      (setq func (lookup-key (current-global-map) key))
      (cond
       ((and (markerp mew-draft-buffer-attach) (mew-draft-in-attach-p))
	(setq func (lookup-key mew-draft-attach-map key)))
       ((and (markerp mew-draft-buffer-header) (mew-draft-in-header-p))
	(setq func (lookup-key mew-draft-header-map key)))
       (t 
	(setq func (lookup-key mew-draft-body-map key))
	)
       )
      )
    (if func
	()
      (setq func (lookup-key (current-global-map) key))
      (if (not (integerp func))
	  ()
	(if (vectorp key) (setq key (mew-subvector key 0 func))
	  (setq key (substring key 0 func)))
	(setq func (lookup-key (current-global-map) key))
	)
      )
    (if func
	(while (keymapp func)
	  (if (vectorp key)
	      (setq key (vconcat key (read-event)))
	    (setq key (concat key (char-to-string (read-event)))))
	  (setq func (lookup-key (current-global-map) key))
	  )
      )
    (if (null func)
	(insert key) ;; just in case
      (setq this-command func)
      (run-hooks 'pre-command-hook)
      (call-interactively this-command))
    ))

(defun mew-draft-null-function ()
  (interactive)
  ())

(defun mew-draft-header-comp ()
  (interactive)
  (let ((func nil))
    (if (mew-draft-on-key-p)
	(mew-draft-key-comp)
      (if (setq func (mew-draft-on-value-p))
	  (funcall (cdr func))
	(tab-to-tab-stop))) ;; default keybinding
    ))

(defun mew-draft-on-key-p ()
  (if (bolp)
      (if (bobp) 
	  t
	(save-excursion
	  (forward-line -1)
	  (if (looking-at ".*,[ \t]?$") nil t)
	  )
	)
    (let ((pos (point)))
      (save-excursion
	(beginning-of-line)
	(if (looking-at mew-lwsp)
	    nil
	  (if (re-search-forward ":" pos t) nil t))
	))
    ))
      
(defun mew-draft-on-value-p ()
  (save-excursion
    (beginning-of-line)
    (while (and (< (point-min) (point))	(looking-at mew-lwsp))
      (forward-line -1)
      )
    (if (looking-at "\\([^:]*:\\)")
	(mew-assoc-match (mew-match 1) mew-field-completion-switch 0)
      nil) ;; what a case reach here?
    ))
      
(defun mew-draft-address-comp ()
  (let ((word nil) (comp nil) (all nil) (len nil))
    (if (null (setq word (mew-delete-backward-char)))
	(tab-to-tab-stop)
      (setq comp (try-completion word mew-alias-alist))
      (setq all (all-completions word mew-alias-alist))
      (setq len (length word))
      (cond
       ((eq comp t)
	(insert (cdr (assoc word mew-alias-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions))
	(if mew-draft-window-config
	    (progn
	      (set-window-configuration mew-draft-window-config)
	      (setq mew-draft-window-config nil)
	      ))
	)
       ((and (char-equal (aref word (1- len)) ?@)
	     (assoc (substring word 0 (1- len)) mew-alias-alist))
	(insert (cdr (assoc (substring word 0 (1- len)) mew-alias-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions))
	(if mew-draft-window-config
	    (progn
	      (set-window-configuration mew-draft-window-config)
	      (setq mew-draft-window-config nil)
	      ))
	)
       ((stringp comp)
	(insert comp)
	(if (> (length all) 1)
	    (progn
	      (or mew-draft-window-config
		  (setq mew-draft-window-config
			(current-window-configuration)))
	      (with-output-to-temp-buffer mew-buffer-completions
		(display-completion-list all))
	      )
	  )
	)
       (t (insert word)
	  (message "No matching alias"))
       )
      )))

(defun mew-draft-folder-comp ()
  (let (word comp all)
    (if (null (setq word (mew-delete-backward-char)))
	(insert "+")
      (setq comp (try-completion word mew-folder-alist))
      (setq all (all-completions word mew-folder-alist))
      (cond
       ((eq comp t)
	(insert (cdr (assoc word mew-folder-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions))
	(if mew-draft-window-config
	    (progn
	      (set-window-configuration mew-draft-window-config)
	      (setq mew-draft-window-config nil)
	      ))
	)
       ((stringp comp)
	(insert comp)
	(if (> (length all) 1)
	    (progn
	      (or mew-draft-window-config
		  (setq mew-draft-window-config
			(current-window-configuration)))
	      (with-output-to-temp-buffer mew-buffer-completions
		(display-completion-list all))
	      )
	  )
	)
       (t (insert word)
	  (message "No matching folder"))
       )
      )))

(defun mew-delete-backward-char ()
  "Delete appropriate preceeding word and return it."
  (let ((case-fold-search t)
	(start nil)
	(end (point))
	;; mew-address-separator == ":, \t\n"
	(regex (concat "[^" mew-address-separator "]")))
    (save-excursion
      ;; find the position of separator.
      (while (and (not (bobp))
		  (string-match regex (mew-buffer-substring 
				       (1- (point)) (point))))
	(forward-char -1))
      (setq start (point))
      (if (= start end)
	  nil ;; cursor is just after separator
	(prog1
	    (mew-buffer-substring start end) ;; return the word
	  (delete-region start end)))
      )
    ))

(defun mew-draft-key-comp ()
  (let (word comp all alist)
    (if (null (setq word (mew-delete-key)))
	(progn
	  (or mew-draft-window-config
	      (setq mew-draft-window-config
		    (current-window-configuration)))
	  (with-output-to-temp-buffer mew-buffer-completions
	    (display-completion-list mew-fields))
	  )
      (setq alist (mew-fiels-make-alist mew-fields)) ;; to make dynamically
      (setq comp (try-completion word alist))
      (setq all (all-completions word alist))
      (cond
       ((eq comp t)
	(insert (cdr (assoc word alist)))
	(insert " ")
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions))
	(if mew-draft-window-config
	    (progn
	      (set-window-configuration mew-draft-window-config)
	      (setq mew-draft-window-config nil)
	      ))
	)
       ((stringp comp)
	(if (equal (length all) 1)
	    (progn
	      (insert (cdr (assoc comp alist)))
	      (insert " ")
	      (and (get-buffer mew-buffer-completions)
		   (kill-buffer mew-buffer-completions))
	      (if mew-draft-window-config
		  (progn
		    (set-window-configuration mew-draft-window-config)
		    (setq mew-draft-window-config nil)
		    )))
	  (insert (capitalize comp))
	  (or mew-draft-window-config
	      (setq mew-draft-window-config
		    (current-window-configuration)))
	  (with-output-to-temp-buffer mew-buffer-completions
	    (display-completion-list all))
	  ))
       (t (insert word)
	  (message "No matching key"))
       ))))

(defun mew-fiels-make-alist (list)
  (mapcar (function 
	   (lambda (x) 
	     (if (string-match "^\\(.*\\):$" x)
		 (cons (capitalize (mew-match 1 x)) (capitalize x)))))
	  list)
  )

(defun mew-delete-key ()
  (let ((pos (point)))
    (beginning-of-line)
    (prog1
	(capitalize (mew-buffer-substring (point) pos))
      (delete-region (point) pos)
      )
    ))

(defun mew-draft-header-domain ()
  (interactive)
  (let ((field (mew-draft-on-value-p)) word)
    (if (or (null field) (not (equal (cdr field) 'mew-draft-address-comp)))
	(lisp-complete-symbol) ;; default key binding
      (setq word (mew-delete-backward-char-at))
      ;; word is strings between @ and cursor
      (cond
       ((equal word nil) ;; @ doesn't exist.
	(if (null mew-mail-domain-list)
	    ()
	  (insert "@")
	  (insert (car mew-mail-domain-list))
	  (and (get-buffer mew-buffer-completions)
	       (kill-buffer mew-buffer-completions))
	  (if mew-draft-window-config
	      (progn
		(set-window-configuration mew-draft-window-config)
		(setq mew-draft-window-config nil)
		))))
       ((equal word t) ;; just after @
	(if mew-mail-domain-list
	    (progn
	      (insert (car mew-mail-domain-list))
	      (and (get-buffer mew-buffer-completions)
		   (kill-buffer mew-buffer-completions))
	      (if mew-draft-window-config
		  (progn
		    (set-window-configuration mew-draft-window-config)
		    (setq mew-draft-window-config nil)
		    )))))
       (t
	(let ((comp nil) (all nil)
	      (dalist (mew-slide-pair mew-mail-domain-list)))
	  (setq comp (try-completion word dalist))
	  (setq all (all-completions word dalist))
	  (cond
	   ;; already completed
	   ((eq comp t) 
	    (insert (cdr (assoc word dalist))) ;; next candidate
	    (and (get-buffer mew-buffer-completions)
		 (kill-buffer mew-buffer-completions))
	    (if mew-draft-window-config
		(progn
		  (set-window-configuration mew-draft-window-config)
		  (setq mew-draft-window-config nil)
		  )))
	   ;; just one candidate
	   ((equal 1 (length all))
	    (insert (car all))
	    (and (get-buffer mew-buffer-completions)
		 (kill-buffer mew-buffer-completions))
	    (if mew-draft-window-config
		(progn
		  (set-window-configuration mew-draft-window-config)
		  (setq mew-draft-window-config nil)
		  ))
	    (message "Sole completion"))
	   ;; one or more candidate
	   ((stringp comp)
	    (insert comp)
	    (if (> (length all) 1)
		(progn
		  (or mew-draft-window-config
		      (setq mew-draft-window-config
			    (current-window-configuration)))
		  (with-output-to-temp-buffer mew-buffer-completions
		    (display-completion-list all))
		  )))
	   ;; no candidate
	   (t (insert word)
	      (message "No matching alias"))
	   )
	  ))
       ))
    ))

(defun mew-delete-backward-char-at ()
  (interactive)
  (let ((case-fold-search t)
	(start nil)
	(end (point))
	(regex (concat "[^" mew-address-separator "]")))
    (save-excursion
      (while (and (not (bobp))
		  (string-match regex
				(mew-buffer-substring (1- (point)) (point))))
	(forward-char -1))
      (if (null (re-search-forward "@" end t))
	  nil ;; @ doesn't exist.
	(setq start (point))
	(if (= start end)
	    t ;; just after @
	  (prog1
	      (mew-buffer-substring start end)
	    (delete-region start end)))
	)
      )
    ))

(defun mew-slide-pair (x)
  (let ((ret nil)
	(first (car x)))
    (cond 
     ((eq x 0) nil)
     ((eq x 1) (cons first first))
     (t
      (while (cdr x)
	(setq ret (cons (cons (nth 0 x) (nth 1 x)) ret))
	(setq x (cdr x))
	)
      (setq ret (cons (cons (car x) first) ret))
      (nreverse ret)
      )
     )
    )
  )

;;
;; undo
;;

(defun mew-draft-undo ()
  (interactive)
  (mew-draft-refresh)
  (let* ((path (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	 (backup-file (concat path "/" mew-draft-backup-file))
	 (syntax-file (concat path "/" mew-draft-syntax-file))
	 (syntax nil))
    (if (not (and (file-exists-p backup-file) (file-exists-p syntax-file)))
	(message "Can't undo")
      (erase-buffer)
      (insert-file-contents backup-file)
      (goto-char (marker-position mew-draft-buffer-header))
      (save-excursion
	(if (not (find-file-read-only syntax-file))
	    ()
	  (goto-char (point-min))
	  (setq syntax (read (current-buffer)))
	  (kill-buffer (current-buffer))
	  ))
      (setq mew-encode-syntax syntax)
      (mew-draft-prepare-attachments)
      (delete-file backup-file)
      (delete-file syntax-file)
      )
    ))

(provide 'mew-draft)
