;;;			    mew-decode.el
;;;
;;;	      Copyright (C) 1996-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: October  2, 1996
;;; Revised: April   21, 1997
;;;

(defconst mew-decode-version "mew-decode.el version 0.12")

(require 'mew)

(defvar mew-file-max-size 100000
  "*The max size of messages. If the size of a message is greater
than mew-file-max-size, Mew asks the user to insert the entire file
or truncate it.")

(defvar mew-header-max-length 50
  "*If the length of a header exceeds this value, 
it is not arranged nor MIME decoded.
See also mew-header-max-depth.")

(defvar mew-field-visible
  '("Subject:" "From:" "To:" "Apparently-To:"
    "Cc:" "Newsgroups:" "Date:" "Reply-To:"
    "Resent-From:" "Resent-To:" "Resent-Cc:"
    "Mime-Version:" "Content-Type:" "Content-Transfer-Encoding:")
  "*Visible fields when a message is displayed in a message mode.")

(defvar mew-field-invisible
  '("Received:" "Return-Path:" "Sender:" "Errors-To:"
    "Resent-Date:" "Message-Id:" "Resent-Message-Id:"
    "Resent-Sender:" "Resent-Reply-To:" "Delivery-Date:")
  "*These fields are hidden when a message is displayed over a message
buffer. Back-scroll makes them visible.")

(defvar mew-field-other-visible t
  "*If *non-nil*, other fields than fields of mew-field-visible and
that of mew-field-invisible are displayed after that of mew-field-visible.
Otherwise, they are hidden over a message buffer. That is, 
fields of mew-field-visible are visible only.")

(defvar mew-rfc822-fields
   '(("To:" . nil)
     ("Cc:" . nil)
     ("From:" . t)
     ("Resent-To:" . nil)
     ("Resent-Cc:" . nil)
     ("Resent-From:" . t)
     ("Reply-To:" . nil)
     ("Subject:" . t))
   "Key and del pair alist for decoding. 
Key is one of RFC822 fields not of MIME fields.
If del is t, linear-white-spaces for folding are removed.
See also mew-mime-fields."
   )

(defvar mew-field-visible-regex nil)
(defvar mew-field-invisible-regex nil)
(defvar mew-field-visible-length 0)
(defvar mew-keyval "^\\([^ \t:]+:\\)[ \t]*\\(.*\\)$")

(defconst mew-ct-txt "Text/Plain")
(defconst mew-ct-msg "Message/Rfc822")
(defconst mew-ct-ext "Message/External-Body")
(defconst mew-ct-mlm "Multipart/Mixed")
(defconst mew-ct-mls "Multipart/Signed")
(defconst mew-ct-mle "Multipart/Encrypted")
(defconst mew-ct-mld "Multipart/Digest")
(defconst mew-ct-ado "Audio/Basic")
(defconst mew-ct-apo "Application/Octet-Stream")
(defconst mew-ct-pgs "application/pgp-signature") ;; xxx case
(defconst mew-ct-pge "application/pgp-encrypted") ;; xxx case
(defconst mew-ct-apk "Application/Pgp-keys")

(defconst mew-type-txt (list mew-ct-txt "charset=us-ascii"))
(defconst mew-type-msg (list mew-ct-msg))
(defconst mew-type-mlm (list mew-ct-mlm))
(defconst mew-type-ado (list mew-ct-ado))
(defconst mew-type-apo (list mew-ct-apo))
(defconst mew-type-apk (list mew-ct-apk))

(defvar mew-prog-mime-decode-switch
  (list (cons mew-b64 '("-b"))
	(cons mew-qp  '("-q"))
	(cons mew-xg  '("-g"))))

(defvar mew-prog-mime-decode-text-switch
  (list (cons mew-b64 '("-b" "-t"))
	(cons mew-qp  '("-q"))
	(cons mew-xg  '("-g" "-t"))))

(defvar mew-decode-multipart-encrypted-switch
  '(("application/pgp-encrypted"  . mew-pgp-decrypt)
    ("application/moss-keys"      . mew-moss-decrypt))
  )

(defvar mew-decode-multipart-signed-switch
  '(("application/pgp-signature"  . mew-pgp-verify)
    ("application/moss-signature" . mew-moss-verify))
  )

(defvar mew-decode-multipart-protocol-prog
  (list 
   (cons "application/pgp-encrypted" mew-prog-pgp)
   (cons "application/pgp-signature" mew-prog-pgp)
   ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; MIME decoder
;;

(defmacro mew-decode-error (error-msg)
  (` (progn (setq mew-decode-error (, error-msg)) (error))))

(defun mew-decode-rfc822-header ()
  ;; Called on the beginning of the header in the narrowed region
  ;; (1) Decode RFC822 fields exclusing MIME fields
  ;; (2) Delete X-Mew: fields
  ;; (3) Arrange decoded-RFC822-fields, mew-mv:, MIME fields in order
  ;; Return whether MIME or not
  (if (not (re-search-forward mew-eoh nil t))
      (mew-decode-error "No end-of-header for RFC822 header"))
  (if (> (count-lines (point-min) (point)) mew-header-max-length)
      (error "Too long header")
    (let ((case-fold-search t)
	  key beg val value attr cnt-hdrs mv back)
      (save-restriction
	(narrow-to-region (point-min) (point))
	(goto-char (point-min))
	(while (not (eobp))
	  (if (not (looking-at mew-keyval))
	      (mew-decode-error "'key: value' pair not found"))
	  (setq key (mew-match 1))
	  (setq val (match-beginning 2))
	  (setq beg (match-beginning 0))
	  (forward-line)
	  (while (looking-at mew-lwsp) (forward-line))
	  (cond
	   ((setq attr (mew-assoc key mew-rfc822-fields 0 t))
	    (forward-char -1)
	    (setq value (mew-buffer-substring val (point)))
	    (delete-region val (point))
	    (insert (mew-header-decode value (cdr attr)))
	    (forward-line))
	   ((string-match key mew-mv:)
	    ;; MIME-Version:
	    (setq mv (buffer-substring beg (point)))
	    (delete-region beg (point)))
	   ((string-match key mew-x-mew:)
	    ;; deleting X-Mew: on the RFC822 header
	    (delete-region beg (point)))
	   ((mew-assoc key mew-mime-fields 0 t)
	    ;; save MIME fields
	    (setq cnt-hdrs (concat cnt-hdrs (buffer-substring beg (point))))
	    (delete-region beg (point)))
	   )
	  ))
      (if mv (insert mv))
      (if (not cnt-hdrs)
	  ()
	(setq back (point))
	(insert cnt-hdrs)
	(goto-char back))
      ;; the beginning of the content header
      (if mv (string-match (concat mew-mv: "[ \t]*" mew-mime-version) mv))
      )
    ))

(defun mew-position (list a)
  (let ((n 0)
	(ret nil))
    (catch 'loop 
      (while list 
	(if (equal (downcase (car (car list))) (downcase a))
	    (throw 'loop (setq ret n)))
	(setq n (1+ n))
	(setq list (cdr list))
	)
      )
    ret
    ))

(defun mew-decode-mime-header (&optional dct)
  ;; Called on the beginning of the content header in the narrowed region
  ;; Return a part syntax after moving the beginning of the content body
  (if (not (re-search-forward mew-eoh))
      (mew-decode-error "No end-of-header for MIME header"))
  (if (> (count-lines (point-min) (point)) mew-header-max-length)
      (error "Too long header")
    (let* ((case-fold-search t)
	   (len (length mew-mime-fields))
	   (vec (make-vector len nil))
	   key val position attr value)
      (aset vec 0 (if dct dct mew-type-txt))
      (save-restriction
	(narrow-to-region (point-min) (point))
	(goto-char (point-min))
	(while (not (eobp))
	  (if (not (looking-at mew-keyval))
	      (mew-decode-error "'key: value' pair not found"))
	  (setq key (mew-match 1))
	  (setq val (match-beginning 2))
	  (forward-line)
	  (while (looking-at mew-lwsp) (forward-line))
	  (if (not (setq position (mew-position mew-mime-fields key)))
	      ()
	    (setq attr (cdr (mew-assoc key mew-mime-fields 0 t)))
	    (setq value (mew-buffer-substring val (point)))
	    (cond
	     ((equal attr 'analyze)
	      (setq value (mew-header-syntax-list value))) ;; list
	     ((equal attr 'extract)
	      (setq value (mew-header-syntax value)))
	     ((equal attr 'decode)
	      (setq value (mew-header-decode value nil))
	      (delete-region val (point))
	      (insert value)
	      (setq value (substring value 0 -1)) ;; delete \n
	      )
	     )
	    (aset vec position value)
	    )
	  ))
      (forward-line) 
      ;; the beginning of the content body
      (vconcat (list 'single (point) nil nil) vec)
      )
    ))

(defun mew-decode-mime (enc &optional text charset)
  ;; Called on the beginning of the content body in the narrow region
  ;; Decode the content body
  ;; Convert text
  (let ((beg (point))
	(switch mew-prog-mime-decode-switch)
	opt file)
    (if (null enc)
	() ;; just in case
      (if text (setq switch mew-prog-mime-decode-text-switch))
      (setq opt (cdr (mew-assoc-match enc switch 0)))
      ;; opt is nil if 7bit, 8bit, and binary
      (if opt
	  (mew-flet 
	   ;; NEVER use call-process-region for privary reasons
	   (if (not (file-exists-p mew-temp-dir))
	       (mew-make-directory mew-temp-dir)) ;; just in case
	   (setq file (make-temp-name mew-temp-file))
	   (write-region beg (point-max) file)
	   (message "") ;; flush the message
	   (delete-region beg (point-max))
	   ;; input is always in ASCII domain
	   (mew-plet
	    (apply (function call-process) mew-prog-mime-decode 
		   file t nil opt)
	    )
	   (if (file-exists-p file) (delete-file file))
	   ))
      )
    (mew-cs-decode-region beg (point-max) charset 'charset)
    ))

;;
;; Kick start function
;;

(defun mew-file-size (file)
  (nth 7 (file-attributes file))
  )

(defun mew-decode (file &optional edit)
  ;; in cache buffer
  (setq mew-decode-error nil)
  ;; must be set mc-flag on Mule 2 for re-search-forward
  (if (and mew-emacs-p mew-mule-p) (setq mc-flag nil))
  (widen)
  (erase-buffer)
  (mew-flet
   (let ((size (mew-file-size file)))
     (if (and (> size mew-file-max-size)
	      (mew-y-or-n-p
	       (format "This mail size is %s. Truncate it? " size)))
	 (setq size mew-file-max-size))
     (insert-file-contents file nil 0 size)
     ))
  (goto-char (point-min))
  ;; Illegal messages may not have end-of-header.
  ;; Truncated messages may not have end-of-header.
  (if (re-search-forward mew-eoh nil t)
      () ;; OK
    (setq mew-decode-error "No end-of-header(null line) in the top level")
    (goto-char (point-max))
    (insert "\n\n"))
  (goto-char (point-min))
  ;; the beginning of the message
  (if mew-debug
      (let ((debug-on-error t))
	(setq mew-decode-syntax
	      (mew-decode-message (mew-decode-syntax-rfc822-head) edit)))
    (condition-case nil
	(setq mew-decode-syntax 
	      (mew-decode-message
	       ;; Call internalform with VIRTUAL content header
	       ;;     CT: message/rfc822 (virtual)
	       ;; 
	       ;;     Header(RFC822 header + content header)
	       ;;
	       ;;     Body(content body)
	       (mew-decode-syntax-rfc822-head) edit))
      (error
       (widen)
       (goto-char (point-min))
       ;; There is certainly end-of-header but just in case.
       (re-search-forward mew-eoh nil t)
       (forward-line)
       ;; the beginning of the body
       (setq mew-decode-syntax (mew-decode-syntax-rfc822))
       ;; min, point-1, point, point-max
       )
      ))
  (setq mew-cache-attribute (mew-cache-attribute-get file))
  )

;;
;; function "m"
;;

(defun mew-decode-message (syntax &optional edit)
  ;; Called on the beginning of the RFC822 header in the narrowed region
  ;; hbeg is certainly the beginning of the VIRTUAL content body(i.e. min).
  ;; hend will have to set to the end of PHYSICAL content header(i.e. end)
  ;; after analyzing the physical content header and body since CD:'s 
  ;; length in the physical content header will change(no need to say
  ;; about the end of the physical content header).
  ;;
  ;;     Content-Type: Message/Rfc822    == virtual content header
  ;;
  ;;(min)Decoded RFC822 fields           == virtual content body
  ;;     MIME-Version: 1.0
  ;;(cur)MIME fields                     == physical content header
  ;;(end)
  ;;     Content-Body                    == physical content body
  ;;(max)
  (let ((mimep (mew-decode-rfc822-header))
	;; the beginning of the physical content header (cur)
	part)
    (cond 
     (mimep ;; MIME
      (save-restriction
	(narrow-to-region (point) (point-max))
	(setq part (mew-decode-singlepart nil 'message edit))
	;; hend is always 1 char smaller than the beginning of 
	;; the physical content body
	(mew-syntax-set-key syntax 'message)
	(mew-syntax-set-end syntax (1- (mew-syntax-get-begin part)))
	(mew-syntax-cat syntax part) ;; the return value
	))
     (t ;; RFC822
      ;; the beginning of the meaningless physical content header
      (if (not (re-search-forward mew-eoh nil t))
	(mew-decode-error "No end-of-header(null line) in RFC822 message"))
      (forward-line)
      ;; the beginning of the BODY(i.e. the physical content body)
      (mew-cs-decode-region (point) (point-max) mew-cs-rfc822-trans)
      (mew-syntax-set-key syntax 'message)
      (mew-syntax-set-end syntax (1- (point)))
      (mew-decode-syntax-rfc822 syntax)
      ;; (point-min), (point) - 1, (point), (point-max)
      )
     )
    ))

;;
;; function "S"
;;

(defun mew-decode-singlepart (&optional dct parent edit)
  ;; Called on the beginning of the content header in the narrowed region
  (let* ((case-fold-search t)
	 (begin (point))
	 (syntax (mew-decode-mime-header dct))
	 ;; the beginning of the content body
	 (ctl (mew-syntax-get-ct syntax))
	 (ct (car ctl))
	 (cte (mew-syntax-get-cte syntax))
	 charset)
    (cond 
     ;; Message/Rfc822, decoding is not required
     ((string-match mew-ct-msg ct)
      (if (null edit)
	  (save-restriction
	    (narrow-to-region (point) (point-max))
	    (if (equal parent 'message)
		;; Rfc822 directly under rfc822 since both his parent 
		;; and his child are rfc822. It must be multipart on Mew!
		;; Otherwise, Mew cannot display the syntax.
		(let ((head mew-encode-syntax-multi-head)
		      (part (mew-decode-message syntax)))
		  ;; begin for multipart syntax is important because
		  ;; the begin will be used by the parent to set hend
		  (mew-syntax-set-begin head (mew-syntax-get-begin part))
		  (mew-syntax-set-end head (point-max))
		  (mew-syntax-cat head part))
	      (mew-decode-message syntax)))
	;; return without analyzing this message
	(mew-syntax-set-end syntax (point-max))
	syntax ;; the return value
	))
     ((string-match mew-ct-ext ct)
      (let* ((access-type (mew-syntax-get-member ctl "access-type"))
	     (func (cdr (mew-assoc access-type mew-ext-include-switch 0 t))))
	(if (not (and func (fboundp func)))
	    (progn
	      ;; probably cte is 7bit
	      (mew-syntax-set-end syntax (point-max))
	      syntax ;; the return value
	      )
	  (save-excursion
	    (goto-char (point-max)) ;; phantom body
	    (funcall func ctl)
	    )
	  (delete-region begin (point))
	  ;; xxx again how about edit?
	  (mew-decode-singlepart)))
      )
     ;; Multipart, decoding is not required
     ((string-match "^Multipart" ct)
      (let* ((protocol (mew-syntax-get-member ctl "protocol"))
	     (switch mew-decode-multipart-protocol-prog) ;; save length
	     (prog (if protocol (cdr (mew-assoc protocol switch 0 t))))
	     (existp (if prog (mew-which prog exec-path))))
	(cond 
	 ((string-match mew-ct-mld ct)
	  ;; semantics into digest
	  (mew-decode-multipart syntax mew-type-msg edit))
	 ((and (string-match mew-ct-mls ct) existp)
	  (mew-decode-multipart-signed syntax edit))
	 ((and (string-match mew-ct-mle ct) existp)
	  (mew-decode-multipart-encrypted syntax edit))
	 (t
	  (mew-decode-multipart syntax nil edit))
	 ))
      )
     ;; Others
     (t  
      (if (null cte) (setq cte "7bit"))
      ;; even if cte is nil, call mew-decode-mime for charset conversion
      (cond 
       ((string-match "^Text/" ct)
	(setq charset (mew-syntax-get-member ctl "charset"))
	;;charset is "us-ascii" if it is nil.
	(mew-decode-mime cte 'text charset))
       ((mew-member-match ct mew-mime-content-type-text-list)
	(mew-decode-mime cte 'text))
       (t
	(mew-decode-mime cte)
	)
       )
      (mew-syntax-set-end syntax (point-max))
      syntax ;; the return value
      )
     )
    ))

;;
;; function "M"
;;

(defun mew-decode-multipart (syntax &optional dct edit)
  (let* ((case-fold-search nil) ;; boundary is case sensitive
	 (ctl (mew-syntax-get-ct syntax))
	 (boundary (regexp-quote (mew-syntax-get-member ctl "boundary")))
	 (parts []) part
	 obound ebound bregex start break)
    (if (null boundary)
	(error "No boundary parameter for multipart"))
    (mew-syntax-set-key syntax 'multi)
    (setq obound (concat "--" boundary))
    (setq ebound (concat "--" boundary "--"))
    (setq bregex (concat "^--" boundary "-?-?$"))
    (if (not (re-search-forward (concat "^" obound "$") nil t))
	(mew-decode-error (format "No first boundary for %s" (car ctl))))
    (forward-line)
    (setq start (point)) ;; the beginning of the part
    (catch 'multipart
      (while 1
	(if (not (re-search-forward bregex nil t))
	    (mew-decode-error (format "No last boundary for %s" (car ctl))))
	(setq break (string= (regexp-quote (mew-match 0)) ebound))
	(forward-line) ;; the beginning of the next part
	(save-excursion
	  (forward-line -1)
	  (beginning-of-line) ;; just in case
	  (forward-char -1) ;; skip the preceeding CRLF
	  ;; the end of the part
	  (save-restriction
	    (narrow-to-region start (point))
	    (goto-char (point-min))
	    ;; the beginning of the part
	    (setq part (mew-decode-singlepart dct nil edit))
	    (setq parts (vconcat parts (vector part)))
	    ))
	(setq start (point)) ;; the beginning of the part
	(if break 
	    (progn
	      (mew-syntax-set-end syntax (point-max)) ;; not necessary
	      (throw 'multipart (vconcat syntax parts)))
	  )
	))
    ))

;;
;; function "P"
;;

(defun mew-decode-multipart-encrypted (syntax &optional edit)
  ;; called in narrowed region
  ;;
  ;;     CT: M/E; proto; bound;
  ;;
  ;;(cur)--bound
  ;;             (the key part)
  ;;     --bound
  ;;             (the encrypted part)
  ;;     --bound--
  (let* ((case-fold-search nil) ;; boundary is case sensitive
	 (ctl (mew-syntax-get-ct syntax))
	 (boundary (regexp-quote (mew-syntax-get-member ctl "boundary")))
	 (protocol (mew-syntax-get-member ctl "protocol"))
	 (switch mew-decode-multipart-encrypted-switch) ;; save length
	 (func (cdr (mew-assoc protocol switch 0 t)))
	 file1 file2 file3 syntax3
	 start result file3result privacy
	 oregex eregex)
    (if (null boundary)
	(error "No boundary parameter for multipart"))
    (if (null protocol)
	(error "No protocol parameter for security multipart"))
    (setq oregex (concat "^--" boundary "$"))
    (setq eregex (concat "^--" boundary "--$"))
    ;;
    (if (not (re-search-forward oregex))
	(mew-decode-error "No first boundary for Multipart/Encrypted"))
    (forward-line) ;; the beginning of the key part
    (setq start (point))
    ;;
    (if (not (re-search-forward oregex))
	(mew-decode-error "No second boundary for Multipart/Encrypted"))
    (beginning-of-line)
    (setq file1 (mew-save-decode-form start (1- (point))))
    (forward-line) ;; the beginning of the encrypted part
    (setq start (point)) 
    ;;
    (if (not (re-search-forward eregex nil t))
	(mew-decode-error "No third boundary for Multipart/Encrypted"))
    (beginning-of-line)
    (setq file2 (mew-save-decode-form start (1- (point))))
    ;;
    (delete-region (point-min) (point-max))
    ;; 
    ;; Call protocol function
    (setq file3result (funcall func file1 file2))
    (setq file3 (nth 0 file3result) result (nth 1 file3result))
    ;;
    (if (null (file-exists-p file3))
	(progn
	  (insert "\n") ;; CT: text/plain; charset=us-ascii
	  (insert "Multipart/Encrypted could not be decrypted.\n"))
      (mew-flet 
       (insert-file-contents file3)
       ;; because of RICH functionality of RFC1847... Gee dirty!
       (mew-decode-crlf-magic)
       ))
    ;; Analyze the decrypted part
    (goto-char (point-min))
    (setq syntax3 (mew-decode-singlepart nil nil edit))
    (setq privacy (mew-syntax-get-privacy syntax3))
    (if (and privacy (not (equal privacy "")))
	(setq result (concat result "\n\t" privacy)))
    (mew-syntax-set-privacy syntax3 result)
    ;; Throw away garbage
    (if (file-exists-p file1) (delete-file file1))
    (if (file-exists-p file2) (delete-file file2))
    (if (file-exists-p file3) (delete-file file3))
    syntax3 ;; return value
    ))

(defun mew-decode-multipart-signed (syntax &optional edit)
  ;; called in narrowed region
  ;;
  ;;     CT: M/S; proto; bound; micalg;
  ;;
  ;;(cur)--bound
  ;;             (the signed part)
  ;;     --bound
  ;;             (the key part)
  ;;     --bound--
  (let* ((case-fold-search nil) ;; boundary is case sensitive
	 (ctl (mew-syntax-get-ct syntax))
	 (boundary (regexp-quote (mew-syntax-get-member ctl "boundary")))
	 (protocol (mew-syntax-get-member ctl "protocol"))
	 (switch mew-decode-multipart-signed-switch) ;; save length
	 (func (cdr (mew-assoc protocol switch 0 t)))
	 file1 file2 syntax3 end1 start2 result privacy
	 oregex eregex)
    (if (null boundary)
	(error "No boundary parameter for multipart"))
    (if (null protocol)
	(error "No protocol parameter for security multipart"))
    (setq oregex (concat "^--" boundary "$"))
    (setq eregex (concat "^--" boundary "--$"))
    ;;
    (if (not (re-search-forward oregex nil t))
	(mew-decode-error "No first boundary for Multipart/Signed"))
    (forward-line)
    ;; the beginning of the signed part
    (delete-region (point-min) (point)) ;; deleting content-header
    (goto-char (point-min)) ;; just in case
    ;;
    (if (not (re-search-forward oregex nil t))
	(mew-decode-error "No second boundary for Multipart/Signed"))
    (beginning-of-line) 
    (setq end1 (1- (point))) ;; the end of the signed part
    (forward-line) ;; the beginning of the key part
    (setq start2 (point)) 
    ;;
    (setq file1 (mew-save-signature-form (point-min) end1))
    ;;
    (if (not (re-search-forward eregex))
	(mew-decode-error "No third boundary for Multipart/Signed"))
    (beginning-of-line) ;; the end of the encrypted part + 1
    (setq file2 (mew-save-decode-form start2 (1- (point))))
    ;;
    (delete-region end1 (point-max))
    ;; Now the signed part only
    ;; Call protocl function
    (setq result (funcall func file1 file2))
    ;; Analyze the signed part
    (goto-char (point-min))
    (setq syntax3 (mew-decode-singlepart nil nil edit))
    (setq privacy (mew-syntax-get-privacy syntax3))
    (if (and privacy (not (equal privacy "")))
	(setq result (concat result "\n\t" privacy)))
    (mew-syntax-set-privacy syntax3 result)
    ;; Throw away garbage
    (if (file-exists-p file1) (delete-file file1))
    (if (file-exists-p file2) (delete-file file2))
    ;;
    syntax3 ;; return value
    ))

(defun mew-save-decode-form (beg end)
  (mew-flet
   (let (syntax file)
     (if (not (file-exists-p mew-temp-dir))
	 (mew-make-directory mew-temp-dir)) ;; just in case
     (setq file (make-temp-name mew-temp-file)) 
     (save-excursion
       (save-restriction
	 (narrow-to-region beg end)
	 (goto-char (point-min))
	 (setq syntax (mew-decode-singlepart))
	 (write-region (mew-syntax-get-begin syntax)
		       (mew-syntax-get-end syntax)
		       file)
	 ))
     file
     )
   ))

(defun mew-decode-crlf-magic ()
  (let ((case-fold-search t)
	(cte "7bit")
	key start match)
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward 
		"^\r?$\\|^Content-Transfer-Encoding:[ \t]*" nil t)
	  (setq key (mew-match 0))
	  (setq start (match-end 0))
	  (if (string-match "^\r?$" key)
	      (progn
		(save-restriction
		  (if (string-match "binary" cte) ;; cte == "binary^M"
		      (narrow-to-region (point-min) (1+ start))
		    (narrow-to-region (point-min) (point-max)))
		  (goto-char (point-min))
		  (while (search-forward "\r\n" nil t)
		    (replace-match "\n" nil t)))
		(throw 'header nil)))
	  (forward-line)
	  (while (looking-at mew-lwsp) (forward-line))
	  (setq match (mew-buffer-substring start (1- (point))))
	  (setq cte (mew-header-syntax match))
	  )))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Header arrange in a message buffer
;;;

(defun mew-make-field-regex (fields)
  (mapconcat (function (lambda (x) (concat "^" x))) fields "\\|")
  )

(defun mew-make-regexes ()
  (setq mew-field-visible-regex 
	(mew-make-field-regex mew-field-visible))
  (setq mew-field-invisible-regex 
	(mew-make-field-regex mew-field-invisible))
  (setq mew-field-visible-length
	(length mew-field-visible))
  )

(defun mew-header-arrange ()
  ;; just in case
  (if (not mew-field-visible-regex)
      (mew-make-regexes))
  (goto-char (point-min))
  (re-search-forward mew-eoh nil t) ;; xxx hack here
  (if (> (count-lines (point-min) (point)) mew-header-max-length)
      (message "Header is too long")
    (let* ((len mew-field-visible-length)
	   (visible-regex mew-field-visible-regex)
	   (invisible-regex mew-field-invisible-regex)
	   (visible (make-vector len nil))
	   (others nil) (key "") (line "") (beg nil) (n nil))
      (save-restriction
	(narrow-to-region (point-min) (point))
	(goto-char (point-min))
	(while (re-search-forward mew-keyval nil t)
	  (setq key (mew-match 1))
	  (setq beg (match-beginning 0))
	  (forward-line)
	  (while (looking-at mew-lwsp) (forward-line))
	  (if (string-match invisible-regex key)
	      ()
	    (setq line (mew-buffer-substring beg (point)))
	    (delete-region beg (point))
	    (if (null (string-match visible-regex key))
		(setq others (concat others line))
	      (setq n (mew-member-match key mew-field-visible))
	      (aset visible n (concat line (aref visible n)))
	      ))
	  ))
      (if (not mew-field-other-visible)
	  (and others (insert others)))
      (recenter 0)
      (setq n 0)
      (while (< n len)
	(if (aref visible n) (insert (aref visible n)))
	(setq n (1+ n))
	)
      (if mew-field-other-visible
	  (and others (insert others)))
      )
    ;; Some nice features for XEmacs.
    (if mew-xemacs-p (highlight-headers (point-min) (point-max) nil))
    )
  (goto-char (point-max))
  )

(provide 'mew-decode)
