;;;			    mew-attach.el
;;;
;;;	      Copyright (C) 1996-1997  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: October 2, 1996
;;; Revised: April   2, 1997
;;;

(defconst mew-attach-version "mew-attach.el version 0.12")

(require 'mew)

(defvar mew-draft-attach-boundary-begin "----------- attachments -----------")
(defvar mew-draft-attach-boundary-end   "--------0-1-2-3-4-5-6-7-8-9--------")
(defvar mew-draft-attach-map nil
  "A region map for attachments in Draft mode")

(if mew-draft-attach-map
    ()
  (setq mew-draft-attach-map (make-keymap))
  (let ((begin ?\0) (end ?\177))
    (while (<= begin end)
      (define-key mew-draft-attach-map 
	(char-to-string begin) 'mew-draft-null-function)
      (setq begin (1+ begin))))
  (define-key mew-draft-attach-map "\C-f" 'mew-attach-forward)
  (define-key mew-draft-attach-map "\C-b" 'mew-attach-backforward)
  (define-key mew-draft-attach-map "\C-n" 'mew-attach-next)
  (define-key mew-draft-attach-map "\C-p" 'mew-attach-previous)
  (define-key mew-draft-attach-map "a"    'mew-attach-audio)
  (define-key mew-draft-attach-map "c"    'mew-attach-copy)
  (define-key mew-draft-attach-map "C"    'mew-attach-charset)
  (define-key mew-draft-attach-map "d"    'mew-attach-delete)
  (define-key mew-draft-attach-map "D"    'mew-attach-description)
  (define-key mew-draft-attach-map "e"    'mew-attach-external-body)
  (define-key mew-draft-attach-map "f"    'mew-attach-find-file)
  (define-key mew-draft-attach-map "F"    'mew-attach-find-new-file)
  (define-key mew-draft-attach-map "l"    'mew-attach-link)
  (define-key mew-draft-attach-map "m"    'mew-attach-multipart)
  (define-key mew-draft-attach-map "T"    'mew-attach-type)
  (define-key mew-draft-attach-map "G"    'mew-attach-gzip64)
  (define-key mew-draft-attach-map "B"    'mew-attach-base64)
  (define-key mew-draft-attach-map "Q"    'mew-attach-quoted-printable)
  (define-key mew-draft-attach-map "S"    'mew-attach-pgp-sign)
  (define-key mew-draft-attach-map "E"    'mew-attach-pgp-enc)
  (define-key mew-draft-attach-map "p"    'mew-attach-pgp-public-key)
  (define-key mew-draft-attach-map "U"    'mew-attach-undo)
  (define-key mew-draft-attach-map "\C-y" 'mew-attach-dummy)
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; goodies
;;

(defmacro mew-attach-p ()
  (` (markerp mew-draft-buffer-attach))
  )

(defun mew-draft-prepare-attachments ()
  "Prepare an attachment region in the bottom of the draft.
To compose a multipart message, you should execute this command first."
  (interactive)
  (save-excursion
    (let* ((mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (cp (concat mimedir (concat "/" mew-draft-coverpage)))
	   (dir (file-name-nondirectory mimedir)))
      (if (mew-attach-p)
	  ()
	(mew-attach-insert-boundary)
	(if (not (file-exists-p mimedir)) (mew-make-directory mimedir))
	(if (not (file-exists-p cp)) (append-to-file 1 1 cp)) ;; touch
	(if (not mew-encode-syntax)
	    (setq mew-encode-syntax (mew-encode-syntax-initial dir)))
	(mew-encode-syntax-print mew-encode-syntax)
	))
    )
  (mew-attach-goto-number 'here '(2))
  (setq buffer-undo-list nil)
  )

(defun mew-attach-insert-boundary ()
   (goto-char (point-max))
   (if (null (bolp))(insert "\n"))
   (if (mew-attach-p)
       (set-marker mew-draft-buffer-attach (point))
     (setq mew-draft-buffer-attach (point-marker)))
   (insert mew-draft-attach-boundary-begin)
   (insert "\n")
   (insert mew-draft-attach-boundary-end)
   (insert "\n")
   (beginning-of-line)
   )

(defun mew-attach-nums ()
  (let ((nums (mew-encode-syntax-number)))
    (if nums
	(mew-split-number nums)
      nil)
    ))

(defun mew-encode-syntax-number ()
  (let ((event last-command-event)
	ret str)
    (if (and mew-icon-p	
	     (or (button-press-event-p event)
		 (button-release-event-p event)))
	(progn
	  (setq str (toolbar-button-help-string last-pressed-toolbar-button))
	  (if (string-match "^<\\([0-9.]+\\)>" str)
	      (setq ret (mew-match 1 str)))
	  ))
    (if ret
	ret
      (save-excursion
	(beginning-of-line)
	(if (looking-at "^.....\\([.0-9]+\\) +")
	    (mew-match 1)
	  nil
	  )
	))
    ))

(defun mew-attach-expand-path (syntax nums)
  "ignore the first dir"
  (let ((path ""))
    (while (cdr nums)
      (setq syntax (aref syntax (+ mew-syntax-magic (1- (car nums)))))
      (setq path (concat path (mew-syntax-get-file syntax)))
      (setq nums (cdr nums))
      )
    path
    ))

(defun mew-attach-line ()
  "Return the line number for the cursor from the beginning of the 
attachments region. 
0 if on the beginning boundary. 
-1 if on the ending boundary."
  (let (ret max)
    (save-excursion
      (beginning-of-line)
      (setq ret (count-lines (marker-position mew-draft-buffer-attach)(point)))
      (goto-char (point-max))
      (beginning-of-line)
      (setq max (count-lines (marker-position mew-draft-buffer-attach)(point)))
      (if (or (= ret max) (= ret (1- max))) -1 ret)
      )
    ))

;; cursor only
(defun mew-attach-line0-1 ()
  (if (mew-attach-p)
      (let ((line (mew-attach-line)))
	(or (= line 0) (= line -1))
	)
    ))

;; cursor only
(defun mew-attach-line01-1 ()
  (if (mew-attach-p)
      (let ((line (mew-attach-line)))
	(or (= line 0) (= line 1) (= line -1))
	)
    ))

;; insert commands
(defun mew-attach-not-line012-1 ()
  (if (mew-attach-p)
      (let ((line (mew-attach-line)))
	(not (or (= line 0) (= line 1) (= line 2) (= line -1)))
	)
    ))

;; remove, find dommands
(defun mew-attach-not-line012-1-dot ()
  (if (mew-attach-p)
      (let ((line (mew-attach-line)))
	(save-excursion
	  (beginning-of-line)
	  (not (or (looking-at "^.....[.0-9]+ +\\. +")
		   (or (= line 0) (= line 1) (= line 2) (= line -1))))
	  ))
    ))
    
;; modify commands
(defun mew-attach-not-line0-1-dot ()
  (if (mew-attach-p)
      (let ((line (mew-attach-line)))
	(save-excursion
	  (beginning-of-line)
	  (not (or (looking-at "^.....[.0-9]+ +\\. +")
		   (= line 0) (= line -1)))
	  ))
    ))

;; corsor
(defun mew-attach-line1-dot ()
  (if (mew-attach-p)
      (let ((line (mew-attach-line)))
	(save-excursion
	  (beginning-of-line)
	  (or (looking-at "^.....[.0-9]+ +\\. +")
	      (= line 1)))
	)
    ))

(defun mew-attach-goto-number (direction nums)
  (let (numreg)
    (setq nums (nreverse nums))
    (cond
     ((equal direction 'next)
      (setq nums (cons (1+ (car nums)) (cdr nums)))
      )
     ((equal direction 'prev)
      (setq nums (cons (1- (car nums)) (cdr nums)))
      )
     ((equal direction 'up)
      (setq nums (cdr nums))
      )
     ((equal direction 'down)
      (setq nums (cons 1 nums))
      )
     (t ())
     )
    (if (null nums)
	(progn
	  (goto-char (marker-position mew-draft-buffer-attach))
	  (re-search-forward (concat "^..... +")))
      (setq numreg (int-to-string (car nums)))
      (setq nums (cdr nums))
      (while nums
	(setq numreg (concat (int-to-string (car nums)) "." numreg))
	(setq nums (cdr nums))
	)
      (setq numreg (regexp-quote numreg))
      (goto-char (marker-position mew-draft-buffer-attach))
      (re-search-forward (concat "^....." numreg " +"))
      )))

(defvar mew-attach-random-filename "0123456789abcdefghijklmnopqrstuvwxwz")

(defun mew-attach-random-filename ()
  (let* ((len (length mew-attach-random-filename))
	 (idx (% (abs (random)) len)))
    (substring mew-attach-random-filename idx (1+ idx))
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; commands
;;

;;
;; cursor commands
;;

(defun mew-attach-forward ()
  "Go to the first subdirectory in attachments."
  (interactive) 
  (if (mew-attach-line0-1)
      (mew-attach-goto-number 'here nil)
    (beginning-of-line) 
    (if (looking-at "^.....[.0-9]* +[^ ]+/")
	(mew-attach-goto-number 'down (mew-attach-nums))
      (mew-attach-goto-number 'here (mew-attach-nums)))
    ))

(defun mew-attach-backforward ()
  "Go to the parent directory in attachments."
  (interactive)
  (if (mew-attach-line0-1)
      (progn
	(goto-char (marker-position mew-draft-buffer-attach))
	(forward-char -1))
    (let ((nums (mew-attach-nums)))
      (if (= (length nums) 1)
	  (mew-attach-goto-number 'here nil)
	(mew-attach-goto-number 'up (mew-attach-nums))))))

(defun mew-attach-previous ()
  "Go to the previous file in the current directory in attachments."
  (interactive)
  (if (mew-attach-line01-1)
      (progn
	(goto-char (marker-position mew-draft-buffer-attach))
	(forward-char -1))
  (let* ((nums (mew-attach-nums))
	 (last (nth (1- (length nums)) nums)))
    (if (= last 1)
	(mew-attach-goto-number 'here (mew-attach-nums))
      (mew-attach-goto-number 'prev (mew-attach-nums))))))


(defun mew-attach-next ()
  "Go to the next file in the current directory in attachments."
  (interactive)
  (cond
   ((mew-attach-line0-1)
    (mew-attach-goto-number 'here nil))
   ((mew-attach-line1-dot)
    (mew-attach-goto-number 'here (mew-attach-nums)))
   (t
    (mew-attach-goto-number 'next (mew-attach-nums)))
   ))

;;
;; remove commands
;;

(defun mew-attach-delete ()
  "Delete this file or this directory in attachments."
  (interactive)
  (if (not (mew-attach-not-line012-1-dot))
      (message "Can't delete here.")
    (let* ((nums (mew-attach-nums))
	   (subdir (mew-attach-expand-path mew-encode-syntax nums))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (syntax (mew-syntax-get-entry mew-encode-syntax nums))
	   (name (mew-syntax-get-file syntax))
	   (ename (if (equal subdir "") name (concat subdir name)))
	   (fullname (expand-file-name ename mimedir))
	   (dirp (string-match ".*/$" name))
	   (msg (if dirp
		    "Delete %s with contained files? "
		  "Delete %s? ")))
      ;; mimedir / {subdir/} name
      (if (null (mew-y-or-n-p (format msg ename)))
	  ()
	(message "Deleting %s ... " ename)
	(if dirp
	    (mew-delete-directory-recursively fullname)
	  (delete-file fullname))
	(message "Deleting %s ... done" ename)
	(setq mew-encode-syntax
	      (mew-syntax-remove-entry mew-encode-syntax nums))
	(mew-encode-syntax-print mew-encode-syntax)
	))
    ))

;;
;; insert commands
;;

(defun mew-attach-multipart ()
  "Create a subdirectory(i.e. multipart) on \".\" in attachments."
  (interactive)
  (if (not (mew-attach-not-line012-1))
      (message "Can't create a sub-multipart here.")
    (let* ((nums (mew-attach-nums))
	   (subdir (mew-attach-expand-path mew-encode-syntax nums))
	   (insert-default-directory nil)
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (dir (mew-attach-random-filename))
	   (cdir nil)
	   (cnt 0) (max (length mew-attach-random-filename)))
      ;; mimedir / {subdir/} dir
      (if (not (equal subdir "")) 
	  (setq mimedir (expand-file-name subdir mimedir)))
      ;; mimedir / dir
      (setq cdir (expand-file-name dir mimedir))
      (while (and (file-exists-p cdir) (< cnt max))
	(setq dir (mew-attach-random-filename))
	(setq cdir (expand-file-name dir mimedir))
	(setq cnt (1+ cnt)))
      (if (file-exists-p cdir)
	  (message "Could not make a directory, sorry")
	(mew-make-directory cdir)
	(setq mew-encode-syntax
	      (mew-syntax-insert-entry
	       mew-encode-syntax 
	       nums
	       (mew-encode-syntax-multi dir mew-type-mlm)))
	(mew-encode-syntax-print mew-encode-syntax))
      )))

(defun mew-attach-link (&optional from to)
  "Link a file with a symbolic link on \".\" in attachments."
  (interactive)
  (if (not (mew-attach-not-line012-1))
      (message "Can't link here.")
    (let* ((doit t)
	   (nums (mew-attach-nums))
	   (subdir (mew-attach-expand-path mew-encode-syntax nums))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (fromfile (or from (mew-input-filename "Link from : ")))
	   (tofile (or to 
		       (mew-input-string 
			"link to %s(%s): " 
			subdir
			(file-name-nondirectory fromfile))))
	   (efile nil))
      (setq efile (if (equal subdir "") tofile (concat subdir tofile)))
      (if (file-exists-p (expand-file-name efile mimedir))
	  (if (null (mew-y-or-n-p (message "File %s exists. Overwrite it? "
					   efile)))
	      (setq doit nil)
	    (delete-file (expand-file-name efile mimedir))))
      (if (null doit)
	  ()
	(make-symbolic-link fromfile (expand-file-name efile mimedir))
	(setq mew-encode-syntax
	      (mew-syntax-insert-entry
	       mew-encode-syntax 
	       nums
	       (mew-encode-syntax-single tofile)))
	(mew-encode-syntax-print mew-encode-syntax)
	))
    ))

(defun mew-attach-copy (&optional from to)
  "Copy a file (via networks) on \".\" in attachments."
  (interactive)
  (if (not (mew-attach-not-line012-1))
      (message "Can't copy here")
    (let* ((nums (mew-attach-nums))
	   (subdir (mew-attach-expand-path mew-encode-syntax nums))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
 	   (fromfile (or from (mew-input-filename "Copy from : ")))
 	   (tofile (or to 
		       (mew-input-string "Copy to %s(%s): " subdir
					 (file-name-nondirectory fromfile))))
	   (efile nil))
      (setq efile (if (equal subdir "") tofile (concat subdir tofile)))
      (if (and (file-exists-p (expand-file-name efile mimedir))
	       (null (mew-y-or-n-p (message "File %s exists. Overwrite it? " 
					    efile))))
	  ()
	(save-excursion
	  (set-buffer (get-buffer-create mew-buffer-tmp))
	  (widen)
	  (erase-buffer)
	  (mew-flet
	   (insert-file-contents fromfile)
	   (write-region (point-min) (point-max)
			 (expand-file-name efile mimedir))
	   ))
	(setq mew-encode-syntax
	      (mew-syntax-insert-entry
	       mew-encode-syntax 
	       nums
	       (mew-encode-syntax-single tofile)))
	(mew-encode-syntax-print mew-encode-syntax)
	))
    ))

(defun mew-attach-find-new-file ()
  "Open a new file into a buffer on \".\" in attachments."
  (interactive)
  (if (not (mew-attach-not-line012-1))
      (message "Can't find a new file here.")
    (let* ((nums (mew-attach-nums))
	   (subdir (mew-attach-expand-path mew-encode-syntax nums))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (file (mew-input-string "Open file as %s(%s): " subdir "a.txt"))
	   ;; xxx random?
	   (efile nil))
      (setq efile (if (equal subdir "") file (concat subdir file)))
      (setq mew-encode-syntax
	    (mew-syntax-insert-entry
	     mew-encode-syntax 
	     nums
	     (mew-encode-syntax-single file (list mew-ct-txt))))
      (mew-encode-syntax-print mew-encode-syntax)
      ;;
      (find-file (expand-file-name efile mimedir))
      ;; buffer switched
      (local-set-key "\C-c\C-q" 'mew-kill-buffer)
      )
    ))

(defun mew-attach-audio ()
  "Sampling voice and insert as audio file on \".\" in attachments"
  (interactive)
  (if (not (mew-attach-not-line012-1))
      (message "Can't attach audio data here.")
    (let* ((nums (mew-attach-nums))
	   (subdir (mew-attach-expand-path mew-encode-syntax nums))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (file (mew-input-string "Create as %s(%s): " subdir "a.au"))
	   ;; xxx random?
	   efile )
      (setq efile (if (equal subdir "") file (concat subdir file)))
      (save-excursion
	(set-buffer mew-buffer-tmp)
	(erase-buffer)
	(condition-case nil
	    (progn
	      (while (not (mew-y-or-n-p "Are you ready? ")) ())
	      (message "Type C-g to finish recording...")
	      (mew-plet
	       (apply (function call-process) mew-prog-audio)
	       ))
	  (quit (message "Type C-g to finish recording... done.")))
	(mew-flet
	 (write-region (point-min) (point-max)
		       (expand-file-name efile mimedir)))
	)
      (kill-buffer mew-buffer-tmp)
      (setq mew-encode-syntax
	    (mew-syntax-insert-entry
	     mew-encode-syntax 
	     nums
	     (mew-encode-syntax-single file mew-type-ado)))
      (mew-encode-syntax-print mew-encode-syntax)
      )
    ))

;;
;; modify commands
;;

(defun mew-attach-undo ()
  "Get mark back in attachments."
  (interactive)
  (if (not (mew-attach-not-line0-1-dot))
      (message "Can't undo encoding here")
    (mew-syntax-clear-marks mew-encode-syntax (mew-attach-nums))
    (mew-encode-syntax-print mew-encode-syntax)
    )
  )

(defun mew-attach-type ()
  "Change the data type(Content-Type:) in attachments."
  (interactive)
  (if (not (mew-attach-not-line0-1-dot))
      (message "Can't change %s here" mew-ct:)
    (let* ((nums (mew-attach-nums))
	   (syntax (mew-syntax-get-entry mew-encode-syntax nums))
	   (file (mew-syntax-get-file syntax))
	   (ctl (mew-syntax-get-ct syntax))
	   (ct (capitalize (car ctl))))
      (setq ct (mew-input-type "Type for %s (%s): " file ct
			       (if (mew-syntax-multipart-p syntax)
				   mew-mime-content-type-multipart-list
				 mew-mime-content-type-list)))
      (setq ctl (list ct))
      (mew-syntax-set-ct syntax ctl)
      (mew-encode-syntax-print mew-encode-syntax)
      )))

(defun mew-attach-charset ()
  "Specify charset for a Text/* object in attachments."
  (interactive)
  (if (null mew-mule-p)
      (message "This commands cannot be used on this Emacs/XEmacs")
    (if (not (mew-attach-not-line0-1-dot))
	(message "Can't specify character set here")
      (let* ((nums (mew-attach-nums))
	     (syntax (mew-syntax-get-entry mew-encode-syntax nums))
	     (file (mew-syntax-get-file syntax))
	     (ctl (mew-syntax-get-ct syntax))
	     (ct (car ctl))
	     (char (mew-syntax-get-member ctl "charset"))
	     charset
	     (case-fold-search t))
	(if (null (string-match "text/" ct))
	    (message "Can't specify character set to %s" ct)
	  (if (null char) (setq char "guess"))
	  (setq charset (mew-input-type "Charset for %s (%s): "
					file char mew-mule-mime-charset))
	  (if (equal charset "guess")
	      (setq ctl (list ct))
	    (setq ctl (list ct (format "charset=%s" charset))))
	  (mew-syntax-set-ct syntax ctl)
	  (mew-encode-syntax-print mew-encode-syntax)
	  )))
    ))

(defun mew-attach-description (&optional cd)
  "Input a description(Content-Description:) in attachments."
  (interactive)
  (if (not (mew-attach-not-line0-1-dot))
      (message "Can't describe here")
    (let* ((nums (mew-attach-nums))
	   (syntax (mew-syntax-get-entry mew-encode-syntax nums))
	   (file (mew-syntax-get-file syntax)))
      (if (null cd)
	  (setq cd (read-string (format "Description (%s): " file) "")))
      (if (equal cd "") (setq cd file))
      (mew-syntax-set-cd syntax cd)
      (mew-encode-syntax-print mew-encode-syntax)
      )
    ))

(defun mew-attach-base64 ()
  "Mark as Base64(B) in attachments."
  (interactive)
  (if (not (mew-attach-not-line0-1-dot))
      (message "Can't encode with base64 here")
    (let* ((nums (mew-attach-nums))
	   (syntax (mew-syntax-get-entry mew-encode-syntax nums))
	   (ctl (mew-syntax-get-ct syntax))
	   (ct (car ctl))
	   (case-fold-search t))
	(if (or (string-match "^Message/" ct) (string-match "^Multipart/" ct))
	    (message "Can't encode with base64 here")
	  (mew-syntax-set-cte syntax mew-b64)
	  (mew-encode-syntax-print mew-encode-syntax)
	  ))
    ))

(defun mew-attach-quoted-printable ()
  "Mark as Quoted-Printable(Q) in attachments"
  (interactive)
  (if (not (mew-attach-not-line0-1-dot))
      (message "Can't encode with quoted-printable here")
    (let* ((nums (mew-attach-nums))
	   (syntax (mew-syntax-get-entry mew-encode-syntax nums))
	   (ctl (mew-syntax-get-ct syntax))
	   (ct (car ctl))
	   (case-fold-search t))
	(if (or (string-match "^Message/" ct) (string-match "^Multipart/" ct))
	    (message "Can't encode with quoted-printable here")
	  (mew-syntax-set-cte syntax mew-qp)
	  (mew-encode-syntax-print mew-encode-syntax)
	  ))
    ))

(defun mew-attach-gzip64 ()
  "Mark as Gzip64(G) in attachments"
  (interactive)
  (if (not (mew-attach-not-line0-1-dot))
      (message "Can't apply gzip here")
    (let* ((nums (mew-attach-nums))
	   (syntax (mew-syntax-get-entry mew-encode-syntax nums))
	   (ctl (mew-syntax-get-ct syntax))
	   (ct (car ctl))
	   (case-fold-search t))
      (if (null (mew-member-match ct mew-mime-content-type-text-list))
	  ;; never use compress to types other than text/plain or 
	  ;; application/postscript. Indeed, compression is not
	  ;; effective to compressed data such as jpeg.
	  (message "Can't apply gzip to %s" ct)
	(mew-syntax-set-cte syntax mew-xg)
	(mew-encode-syntax-print mew-encode-syntax)
	))
    ))

(defun mew-attach-pgp-sign ()
  "Mark as PGP sign(PS) in attachments."
  (interactive)
  (if (not (mew-attach-not-line0-1-dot))
      (message "Can't PGP sign here")
    (if (null (mew-which mew-prog-pgp exec-path))
	(message "PGP doesn't exist")
      (let* ((nums (mew-attach-nums))
	     (syntax (mew-syntax-get-entry mew-encode-syntax nums))
	     (privacy (mew-syntax-get-privacy syntax)))
	(if (>= (length privacy) 2)
	    (message "Too many marks")
	  (mew-syntax-set-privacy 
	   syntax (append privacy (list (list mew-ct-mls mew-ct-pgs))))
	  (mew-encode-syntax-print mew-encode-syntax)
	  ))
      )))

(defun mew-attach-pgp-enc ()
  "Mark as PGP encrypt in attachment. Input to whom you 
encrypt this part(PE)."
  (interactive)
  (if (not (mew-attach-not-line0-1-dot))
      (message "Can't PGP encrypt here")
    (if (null (mew-which mew-prog-pgp exec-path))
	(message "PGP doesn't exist")
      (let* ((nums (mew-attach-nums))
	     (syntax (mew-syntax-get-entry mew-encode-syntax nums))
	     (privacy (mew-syntax-get-privacy syntax))
	     (towhom (car (mew-header-address-collect '("To:")))))
	(if (>= (length privacy) 2)
	    (message "Too many marks")
	  ;; ask towhom before set privacy for C-g
	  (if towhom
	      (setq towhom (mew-input-address2 "To (%s): " towhom))
	    (setq towhom (mew-input-address "To:")))
	  (mew-syntax-set-privacy 
	   syntax (append privacy (list (list mew-ct-mle mew-ct-pge))))
	  (mew-syntax-set-decrypters syntax towhom)
	  (mew-encode-syntax-print mew-encode-syntax)
	  ))
      )))

;;
;; find commands
;;

(defun mew-attach-find-file ()
  "Open this file into a buffer in attachments."
  (interactive)
  (if (not (mew-attach-not-line012-1-dot))
      (message "Can't find a file here.")
    (let* ((nums (mew-attach-nums))
	   (subdir (mew-attach-expand-path mew-encode-syntax nums))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (syntax (mew-syntax-get-entry mew-encode-syntax nums))	   
	   (file (mew-syntax-get-file syntax))
	   efile)
      (setq efile (if (equal subdir "") file (concat subdir file)))
      (find-file (expand-file-name efile mimedir))
      ;; buffer switched
      (local-set-key "\C-c\C-q" 'mew-kill-buffer)
      )))

;;

(defun mew-attach-dummy ()
  "A dummy function to clear side-effect in attachments."
  (interactive)
  ()
  )

(provide 'mew-attach)
